using UnityEngine;
using UnityEngine.Rendering;

#if UNITY_RENDER_PIPELINE_URP
using UnityEngine.Rendering.Universal;
#endif


[ExecuteAlways]
[DisallowMultipleComponent]
public class URPPlanarReflection : MonoBehaviour
{
#if UNITY_RENDER_PIPELINE_URP
    [Header("Reflection")]
    public LayerMask reflectLayers = -1;
    public int textureSize = 512;
    public float clipPlaneOffset = 0.07f;

    static bool s_Rendering;
    Camera m_ReflectCamera;
    RenderTexture m_ReflectionRT;

    void OnEnable()
    {
        RenderPipelineManager.beginCameraRendering += BeginCameraRendering;
    }

    void OnDisable()
    {
        RenderPipelineManager.beginCameraRendering -= BeginCameraRendering;

        if (m_ReflectCamera != null)
        {
            if (Application.isPlaying) Destroy(m_ReflectCamera.gameObject);
            else DestroyImmediate(m_ReflectCamera.gameObject);
            m_ReflectCamera = null;
        }

        if (m_ReflectionRT != null)
        {
            if (Application.isPlaying) Destroy(m_ReflectionRT);
            else DestroyImmediate(m_ReflectionRT);
            m_ReflectionRT = null;
        }
    }

    void EnsureResources(Camera src)
    {
        if (m_ReflectionRT == null || m_ReflectionRT.width != textureSize)
        {
            if (m_ReflectionRT != null)
            {
                if (Application.isPlaying) Destroy(m_ReflectionRT);
                else DestroyImmediate(m_ReflectionRT);
            }
            m_ReflectionRT = new RenderTexture(textureSize, textureSize, 16, RenderTextureFormat.ARGB32);
            m_ReflectionRT.name = "__URPPlanarReflectionRT";
            m_ReflectionRT.hideFlags = HideFlags.DontSave;
        }

        if (m_ReflectCamera == null)
        {
            var go = new GameObject("URP Reflection Camera");
            go.hideFlags = HideFlags.HideAndDontSave;
            m_ReflectCamera = go.AddComponent<Camera>();
            m_ReflectCamera.enabled = false;

            // URPp̒ǉf[^iURPɂƂj
            var add = go.AddComponent<UniversalAdditionalCameraData>();
            add.renderShadows = false;
            add.requiresColorOption = CameraOverrideOption.Off;
            add.requiresDepthOption = CameraOverrideOption.Off;
        }
    }

    void BeginCameraRendering(ScriptableRenderContext context, Camera cam)
    {

if (s_Rendering) { Debug.Log("skip because s_Rendering==true"); return; }
        if (s_Rendering) return;
        if (cam == null) return;

        // SceneView܂߂Ȃ炱̂܂܁DGameȂ cam.cameraType ōi
        // if (cam.cameraType != CameraType.Game) return;

        // ˃JgōċAȂ
        if (m_ReflectCamera != null && cam == m_ReflectCamera) return;

        EnsureResources(cam);

        var rend = GetComponent<Renderer>();
        if (rend == null || rend.sharedMaterial == null) return;

        s_Rendering = true;
        try
        {
            // ˕
            Vector3 pos = transform.position;
            Vector3 normal = transform.up;

            // ˍs
            float d = -Vector3.Dot(normal, pos) - clipPlaneOffset;
            Vector4 plane = new Vector4(normal.x, normal.y, normal.z, d);

            Matrix4x4 reflection = Matrix4x4.zero;
            CalculateReflectionMatrix(ref reflection, plane);

            Vector3 oldpos = cam.transform.position;
            Vector3 newpos = reflection.MultiplyPoint(oldpos);

            // JݒRs[
            m_ReflectCamera.CopyFrom(cam);
            m_ReflectCamera.targetTexture = m_ReflectionRT;
            m_ReflectCamera.cullingMask = ~(1 << 4) & reflectLayers.value; // water layerO

            // view/proj
            m_ReflectCamera.worldToCameraMatrix = cam.worldToCameraMatrix * reflection;

            Vector4 clipPlane = CameraSpacePlane(m_ReflectCamera, pos, normal, 1.0f);
            Matrix4x4 proj = cam.projectionMatrix;
            proj = CalculateObliqueMatrix(proj, clipPlane);
            m_ReflectCamera.projectionMatrix = proj;

            m_ReflectCamera.transform.position = newpos;
            Vector3 euler = cam.transform.eulerAngles;
            m_ReflectCamera.transform.eulerAngles = new Vector3(-euler.x, euler.y, euler.z);

            // ]JO
            bool oldCulling = GL.invertCulling;
            try
            {
                GL.invertCulling = !oldCulling;

                #pragma warning disable CS0618
                UniversalRenderPipeline.RenderSingleCamera(context, m_ReflectCamera);
                #pragma warning restore CS0618
            }
            finally
            {
                GL.invertCulling = oldCulling;
                s_Rendering = false;
            }

            #pragma warning disable CS0618
            UniversalRenderPipeline.RenderSingleCamera(context, m_ReflectCamera);
            #pragma warning restore CS0618

            GL.invertCulling = oldCulling;
            rend.sharedMaterial.SetTexture("_ReflectionTex", m_ReflectionRT);
        }
        finally
        {
            s_Rendering = false;
        }
    }

    static Vector4 CameraSpacePlane(Camera cam, Vector3 pos, Vector3 normal, float sideSign)
    {
        Vector3 offsetPos = pos + normal * 0.07f;
        Matrix4x4 m = cam.worldToCameraMatrix;
        Vector3 cpos = m.MultiplyPoint(offsetPos);
        Vector3 cnormal = m.MultiplyVector(normal).normalized * sideSign;
        return new Vector4(cnormal.x, cnormal.y, cnormal.z, -Vector3.Dot(cpos, cnormal));
    }

    static void CalculateReflectionMatrix(ref Matrix4x4 reflectionMat, Vector4 plane)
    {
        reflectionMat.m00 = (1F - 2F * plane[0] * plane[0]);
        reflectionMat.m01 = (-2F * plane[0] * plane[1]);
        reflectionMat.m02 = (-2F * plane[0] * plane[2]);
        reflectionMat.m03 = (-2F * plane[3] * plane[0]);

        reflectionMat.m10 = (-2F * plane[1] * plane[0]);
        reflectionMat.m11 = (1F - 2F * plane[1] * plane[1]);
        reflectionMat.m12 = (-2F * plane[1] * plane[2]);
        reflectionMat.m13 = (-2F * plane[3] * plane[1]);

        reflectionMat.m20 = (-2F * plane[2] * plane[0]);
        reflectionMat.m21 = (-2F * plane[2] * plane[1]);
        reflectionMat.m22 = (1F - 2F * plane[2] * plane[2]);
        reflectionMat.m23 = (-2F * plane[3] * plane[2]);

        reflectionMat.m30 = 0F;
        reflectionMat.m31 = 0F;
        reflectionMat.m32 = 0F;
        reflectionMat.m33 = 1F;
    }

    static Matrix4x4 CalculateObliqueMatrix(Matrix4x4 projection, Vector4 clipPlane)
    {
        Vector4 q = projection.inverse * new Vector4(
            Sign(clipPlane.x),
            Sign(clipPlane.y),
            1.0f,
            1.0f
        );
        Vector4 c = clipPlane * (2.0F / Vector4.Dot(clipPlane, q));

        projection[2] = c.x - projection[3];
        projection[6] = c.y - projection[7];
        projection[10] = c.z - projection[11];
        projection[14] = c.w - projection[15];

        return projection;
    }

    static float Sign(float a) => a > 0.0f ? 1.0f : (a < 0.0f ? -1.0f : 0.0f);
    
#else     // UNITY_RENDER_PIPELINE_URP
    // BRP̏ꍇ͉Ȃ
    void OnEnable() { }
    void OnDisable() { }
    void OnDestroy() { }
#endif    // UNITY_RENDER_PIPELINE_URP
}
