;;; ======================================================================
;;; wx_win.clp
;;; Base wxWindow class
;;; ======================================================================

;;; ----------------------------------------------------------------------
;;; CLASS:
;;;	wxWINDOW
;;; SUPERCLASS:
;;;	USER
;;; DESCRIPTION:
;;;	This is the basic window class from which all other window
;;;	classes inherit properties. It is an abstract class, i.e.
;;;	no instances can be created of it. 
;;; ----------------------------------------------------------------------

(defclass MAIN::wxWindow (is-a wxEvtHandler)
	(role abstract)
        (slot event-handler
                (create-accessor read-write)
		(visibility public)
		(default nil))
	(slot parent
		(create-accessor read-write)
		(visibility public)
		(default nil))
	(slot title
		(create-accessor read-write)
		(access initialize-only)
		(visibility public)
		(type STRING)
		(default "No Title"))
	(slot window-name
		(create-accessor read-write)
		(access initialize-only)
		(visibility public)
		(type STRING)
		(default "window"))
	(slot x
;		(create-accessor read-write)
;		(access initialize-only)
		(visibility public)
		(type INTEGER)
;		(range -1 ?VARIABLE)
		(default -1))
	(slot y
;		(create-accessor read-write)
;		(access initialize-only)
		(visibility public)
		(type INTEGER)
;		(range -1 ?VARIABLE)
		(default -1))
	(slot width
;		(create-accessor read-write)
;		(access initialize-only)
		(visibility public)
		(type INTEGER)
;		(range -1 9000)
		(default -1))
	(slot height
;		(create-accessor read-write)
;		(access initialize-only)
		(visibility public)
		(type INTEGER)
;		(range -1 9000)
		(default -1))
	(slot style
		(create-accessor read-write)
		(access initialize-only)
		(visibility public)
		(default "")))

;;; ======================================================================
;;; DEFMESSAGE-HANDLERS
;;; ======================================================================

;;; Do some common window initialisation
(defmessage-handler MAIN::wxWindow win-init primary ()
 (dynamic-put event-handler ?self)
)

(defmessage-handler MAIN::wxWindow delete before ()
;  (printout t "*** wxWindow delete" crlf)
  (if (and (neq ?self:pending-delete TRUE)
           (eq 1 (wxclips-object-exists ?self:id))
      )
    then
    (instance-table-delete-entry ?self:id)
    (dynamic-put pending-delete TRUE)
    (window-delete ?self:id)
  )
  (return TRUE)
)

;(defmessage-handler MAIN::wxWindow remove-event-handlers ()
;        (return TRUE)
;)

(defmessage-handler MAIN::wxWindow show primary (?show)
	(return (long-to-bool (window-show ?self:id (bool-to-long ?show))))
)

(defmessage-handler MAIN::wxWindow enable primary (?enable)
	(return (long-to-bool (window-enable ?self:id (bool-to-long ?enable))))
)

(defmessage-handler MAIN::wxWindow centre primary (?orient)
	(return (long-to-bool (window-centre ?self:id ?orient)))
)

(defmessage-handler MAIN::wxWindow fit primary ()
  (return (long-to-bool (window-fit ?self:id)))
)

(defmessage-handler MAIN::wxWindow set-size primary (?x ?y ?width ?height)
  (dynamic-put x ?x)
  (dynamic-put y ?y)
  (dynamic-put width ?width)
  (dynamic-put height ?height)
  (return (long-to-bool (window-set-size ?self:id ?x ?y ?width ?height)))
)

(defmessage-handler MAIN::wxWindow set-client-size primary (?width ?height)
  (return (long-to-bool (window-set-client-size ?self:id ?width ?height)))
)

(defmessage-handler MAIN::wxWindow get-client-width primary ()
  (return (window-get-client-width ?self:id ))
)

(defmessage-handler MAIN::wxWindow get-client-height primary ()
  (return (window-get-client-height ?self:id ))
)

(defmessage-handler MAIN::wxWindow get-x primary ()
  (if (valid-id ?self:id) then (dynamic-put x (window-get-x ?self:id)))
  (return ?self:x)
)

(defmessage-handler MAIN::wxWindow get-y primary ()
  (if (valid-id ?self:id) then (dynamic-put y (window-get-y ?self:id)))
  (return ?self:y)
)

(defmessage-handler MAIN::wxWindow get-width primary ()
  (if (valid-id ?self:id) then (dynamic-put width (window-get-width ?self:id)))
  (return ?self:width)
)

(defmessage-handler MAIN::wxWindow get-height primary ()
  (if (valid-id ?self:id) then (dynamic-put height (window-get-height ?self:id)))
  (return ?self:height)
)

(defmessage-handler MAIN::wxWindow put-x primary (?x)
  (dynamic-put x ?x)
  (if (valid-id ?self:id) then (window-set-size ?self:id ?x ?self:y -1 -1))
)

(defmessage-handler MAIN::wxWindow put-y primary (?y)
  (dynamic-put y ?y)
  (if (valid-id ?self:id) then (window-set-size ?self:id ?self:x ?y -1 -1))
)

(defmessage-handler MAIN::wxWindow put-width primary (?width)
  (dynamic-put width ?width)
  (if (valid-id ?self:id) then (window-set-size ?self:id -1 -1 ?width ?self:height))
)

(defmessage-handler MAIN::wxWindow put-height primary (?height)
  (dynamic-put height ?height)
  (if (valid-id ?self:id) then (window-set-size ?self:id -1 -1 ?self:width ?height))
)

(defmessage-handler MAIN::wxWindow make-modal primary (?flag)
  (return (long-to-bool (window-make-modal ?self:id (bool-to-long ?flag))))
)

(defmessage-handler MAIN::wxWindow popup-menu primary (?menu ?x ?y)
  (return (long-to-bool (window-popup-menu ?self:id (send ?menu get-id) ?x ?y)))
)

(defmessage-handler MAIN::wxWindow set-cursor primary (?cursor)
  (return (long-to-bool (window-set-cursor ?self:id (send ?cursor get-id))))
)

(defmessage-handler MAIN::wxWindow set-focus primary ()
  (return (long-to-bool (window-set-focus ?self:id)))
)

(defmessage-handler MAIN::wxWindow refresh primary ($?args)
  (bind ?erase-background (nth$ 1 ?args))
  (bind ?x (nth$ 2 ?args))
  (bind ?y (nth$ 3 ?args))
  (bind ?width (nth$ 4 ?args))
  (bind ?height (nth$ 5 ?args))
  
  (if (eq ?erase-background nil) then (bind ?erase-background 1))
  (if (eq ?x nil) then (bind ?x -1))
  (if (eq ?y nil) then (bind ?y -1))
  (if (eq ?width nil) then (bind ?width -1))
  (if (eq ?height nil) then (bind ?height -1))

  (return (long-to-bool (window-refresh ?self:id ?erase-background ?x ?y ?width ?height)))
)

(defmessage-handler MAIN::wxWindow find-window-by-name primary (?name)
  (bind ?child-id (find-window-by-name ?name ?self:id))
  (if (eq ?child-id 0) then
    (return nil))
  (return (instance-table-get-instance ?child-id))
)

(defmessage-handler MAIN::wxWindow find-window-by-label primary (?label)
  (bind ?child-id (find-window-by-label ?label ?self:id))
  (if (eq ?child-id 0) then
    (return nil))
  (return (instance-table-get-instance ?child-id))
)

(defmessage-handler MAIN::wxWindow close primary ($?args)
  (bind ?force (nth$ 1 ?args))
  (if (eq ?force nil) then (bind ?force 0))

  (return (long-to-bool (window-close ?self:id ?force)))
)

;;; OnClose callback
(deffunction gui-window-on-close (?id)
  (bind ?inst (instance-table-get-instance ?id))
  (if (neq ?inst "") then (return (bool-to-long (send (send ?inst get-event-handler) on-close)))
    else (return 0)
  )
 )

;;; OnSize callback
(deffunction gui-window-on-size (?id ?w ?h)
  (bind ?inst (instance-table-get-instance ?id))
  (if (neq ?inst "") then (send (send ?inst get-event-handler) on-size ?w ?h))
 )

;;; OnEvent callback
(deffunction gui-window-on-event (?id ?event)
  (bind ?inst (instance-table-get-instance ?id))
  (if (neq ?inst "") then
    (send [wxglobal-mouse-event] put-id ?event)
    (send (send ?inst get-event-handler) on-event [wxglobal-mouse-event])
  )
 )

;;; OnPaint callback
(deffunction gui-window-on-paint (?id)
  (bind ?inst (instance-table-get-instance ?id))
  (if (neq ?inst "") then (send (send ?inst get-event-handler) on-paint))
 )

;;; OnChar callback
(deffunction gui-window-on-char (?id ?key ?event)
  (bind ?inst (instance-table-get-instance ?id))
  (if (neq ?inst "") then
    (send [wxglobal-key-event] put-id ?event)
    (send [wxglobal-key-event] put-keycode ?key)
    (send (send ?inst get-event-handler) on-char [wxglobal-key-event])
   )
 )

(defmessage-handler MAIN::wxWindow add-event-handlers primary ()
  (call-next-handler)

  (window-add-callback ?self:id OnClose     gui-window-on-close)
  (window-add-callback ?self:id OnSize      gui-window-on-size)
  (window-add-callback ?self:id OnEvent     gui-window-on-event)
  (window-add-callback ?self:id OnPaint     gui-window-on-paint)
  (window-add-callback ?self:id OnChar      gui-window-on-char)
)
