;;; Utilities for printing a file using Windows Notepad.
;;; (notepad-print ?filename)
;;; Filename MUST:
;;; 1) Be a FULL pathname
;;; 2) Have the first letter of the filename capitalized, the rest
;;;    in lower case.
;;; 3) Have an extension.
;;; E.g.
;;; (notepad-print "c:\\temp\\Myfile.txt")
;;;
;;; See (notepad-print-test) below.

;;; Get the path only, given a full filename
;;; E.g. c:\thing\test.txt -> c:\thing
;;;
;;; CAVEATS:
;;; (1) It's possible that the Win 3.1 notepad implementation is
;;;     different (e.g. using a different Print menu command id).
;;;     This is tested on Windows 95.
;;; (2) If Win 3.1 displays filenames in Notepad differently,
;;;     you may need to supply an alternative make-window-title...
;;;     function. You can test for operating system as shown below.
;;; (3) This needs wxCLIPS with the new hwnd-send-message function.
;;; (4) You may want to add a hwnd-iconize statement for neatness
;;;     (so you don't see notepad at work) and also a modeless dialog
;;;     asking the user to wait.

(deffunction get-path (?filename)
 (bind ?pos (length ?filename))
 (bind ?found FALSE)
 (while (and (> ?pos 1) (neq ?found TRUE))
  (bind ?char (sub-string ?pos ?pos ?filename))
  ; Find the first backslash from the end
  (if (eq ?char "\\") then
   (bind ?found TRUE)
  else
   (bind ?pos (- ?pos 1)))
 )
 (if (eq ?found TRUE) then
  (if (> ?pos 1) then (return (sub-string 1 (- ?pos 1) ?filename))
   else (return ""))
 else (return ""))
)

;;; Get the filename only, given a full path
;;; E.g. c:\thing\test.txt -> test.txt

(deffunction get-file (?filename)
 (bind ?pos (length ?filename))
 (bind ?found FALSE)
 (while (and (> ?pos 1) (neq ?found TRUE))
  (bind ?char (sub-string ?pos ?pos ?filename))
  ; Find the first backslash from the end
  (if (eq ?char "\\") then
   (bind ?found TRUE)
  else
   (bind ?pos (- ?pos 1)))
 )
 (if (eq ?found TRUE) then
  (if (> ?pos 1) then (return (sub-string (+ ?pos 1) (length ?filename) ?filename))
   else (return ?filename))
 else (return ?filename))
)

;;; Strip extension
(deffunction strip-extension (?filename)
 (bind ?pos (length ?filename))
 (bind ?found FALSE)
 (while (and (> ?pos 1) (neq ?found TRUE))
  (bind ?char (sub-string ?pos ?pos ?filename))
  ; Find the first dot from the end
  (if (eq ?char ".") then
   (bind ?found TRUE)
  else
   (bind ?pos (- ?pos 1)))
 )
 (if (eq ?found TRUE) then
  (if (> ?pos 1) then (return (sub-string 1 (- ?pos 1) ?filename))
   else (return ?filename))
 else (return ?filename))
)

; In Windows 95, the title of notepad is e.g.
; "Myfile - Notepad".
; So you need to strip off the extension and path,
; capitalise the first letter, make the rest lower
; case. For simplicity's sake, let's
; assume the filename is in the right case:
; MAKE SURE THIS IS TRUE IN YOUR PROGRAM!
; And remember to make the filename a FULL PATH.
; So pass it e.g. c:\thing\Myfile.txt

(deffunction make-window-title-95 (?file)
  (return (get-file (strip-extension ?file)))
)

(deffunction make-window-title (?file)
  (if (eq (get-os-version) "wxWIN95") then
    (return (make-window-title-95 ?file))
   else
    ;;; If this is different for Win 3.1, insert a different
    ;;; function name!!!
    (return (make-window-title-95 ?file))
  )
)

; 14 is the identifier to get notepad to print.
; The file MUST have an extension or notepad gets confused.
(deffunction notepad-print (?file)
  (execute (str-cat "notepad.exe " ?file))
  (bind ?wnd (hwnd-find (str-cat (make-window-title ?file) " - Notepad")))

  (if (eq ?wnd 0) then (return FALSE)
    else
     ; This sends ID_PRINT (14) as a menu WM_COMMAND (273).
     ; The print menu ID was ascertained by using Watcom 'Spy'.
     (hwnd-send-message ?wnd 273 14 0)
     (hwnd-quit ?wnd)
     (return TRUE)
  )
)

(deffunction notepad-print-test ()
  (notepad-print "c:\\temp\\Print.clp")
)