;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; LaTeX document-writing CLIPS code.
;;; Call (write-all ?card-id ?filename) from a suitable event handler.
;;; Originally by JKK; modified by JACS (10/7/94) to support Tex2RTF
;;; and to give user feedback under Windows (not possible under X)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; BUGS: 
;;  1/ LaTeX special characters (_, &, $, %) are not given the appropriate
;;     escape character (\).
;;     CURED, JACS 16/7/94
;;  2/ The first attribute (apart from 'type') of nodes & arcs is assumed to
;;     be the name or label, and is used to identify the node/arc.
;;  3/ If the card is wider than 900 points, the diagram is scaled down. This
;;     is an arbitrary limit. (A default card is 600 points wide).
;;  4/ It's possible to create \subsubsubsections (or even deeper), which
;;     LaTeX does not support. 
;;
;; Other files needed: utils1.clp


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Global variables
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defglobal 
  ?*leftmost-coordinate* = 10000
  ?*leftmost-node-image* = FALSE
  ?*node-images-written* = (mv-append)
  ?*to-list* = (mv-append)
  ?*from-list* = (mv-append)
  ?*expansion-cards* = (mv-append)
  ?*status-dialog* = 0
  ?*status-message* = 0
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Forward declarations
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(deffunction write-card (?one ?two))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Utility functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(deffunction find-arc-from-id (?card ?id)
  (bind ?arc (get-first-card-arc ?card))
  (while (neq ?arc -1)
    (if (eq ?id ?arc)
     then (return ?arc)
     else (bind ?arc (get-next-card-arc))
    )
  )
  (return FALSE) ;; arc not found
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Conversion from normal text to Latex-compatible text (e.g. ampersand conversion)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(deffunction convert-to-latex (?text)
 (bind ?out-string "")
 (bind ?len (length$ ?text))
 (bind ?i 1)
 (while (<= ?i ?len) do
  (bind ?char (sub-string ?i ?i ?text))
  (switch ?char
    (case "&" then
     (bind ?out-string (str-cat ?out-string "\\&"))
    )
    (case "%" then
     (bind ?out-string (str-cat ?out-string "\\%"))
    )
    (case "_" then
     (bind ?out-string (str-cat ?out-string "\\_"))
    )
    (case "$" then
     (bind ?out-string (str-cat ?out-string "\\$"))
    )
    (default 
     (bind ?out-string (str-cat ?out-string ?char))
    )
  )
  (bind ?i (+ ?i 1))
 )
 (return ?out-string)
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; User feedback (Windows only)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(deffunction begin-status-dialog ()
 (if (eq (get-platform) "Windows 3.1") then
  (bind ?*status-dialog* (dialog-box-create 0 "Creating Report" 0))
  (bind ?*status-message* (message-create ?*status-dialog* "Creating report, please wait..."))
  (window-fit ?*status-dialog*)
  (window-show ?*status-dialog* 1)
  (yield)
 )
)

(deffunction end-status-dialog ()
 (if (eq (get-platform) "Windows 3.1") then
  (window-show ?*status-dialog* 0)
  (window-delete ?*status-dialog*)
  (bind ?*status-dialog* 0)
  (yield)
 )
)

(deffunction set-status-message (?text)
 (if (eq (get-platform) "Windows 3.1") then
  (panel-item-set-label ?*status-message* ?text)
  (yield)
 )
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Main functions - the top level functions appear last
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(deffunction write-io (?card ?filename ?attribute ?list)
  (yield)
  (printout ?filename "\\item {\\bf " ?attribute ":} ")
  (bind ?counter 1)
  (bind ?length (length ?list))
  (while (<= ?counter ?length)  
    (bind ?arc (nth ?counter ?list))
    (bind ?name-attrib (get-first-object-attribute ?card ?arc))
    (if (eq ?name-attrib "type") 
     then (bind ?name-attrib (get-next-object-attribute))
    )    
    (bind ?arc-name (convert-to-latex (get-object-string-attribute ?card ?arc ?name-attrib)))
    (if (> ?counter 1) then (printout ?filename ", "))
    (if (neq ?arc-name "") ;; i.e. if an attribute exists and is defined
     then (printout ?filename (convert-to-latex ?arc-name))
     else (printout ?filename (get-object-string-attribute ?card ?arc "type"))
    )
    (bind ?counter (+ ?counter 1))
  )
  (printout ?filename crlf)
)

(deffunction write-arcs (?card ?filename ?list)
  (yield)
  (bind ?counter 1)
  (bind ?length (length ?list))
  (while (<= ?counter ?length)  
    (bind ?arc (nth ?counter ?list))
    (bind ?name-attrib (get-first-object-attribute ?card ?arc))
    (if (eq ?name-attrib "type") 
     then (bind ?name-attrib (get-next-object-attribute))
    )    
    (bind ?arc-name (convert-to-latex (get-object-string-attribute ?card ?arc ?name-attrib)))
    (if (neq ?arc-name "") ;; i.e. if the "name" attribute exists and is defined
     then (printout ?filename "\\item {\\em " (convert-to-latex ?arc-name) "}" crlf)
     else (printout ?filename "\\item {\\em " Arc "}" crlf)
    )
    (printout ?filename "\\begin{itemize}" crlf)
    (bind ?attribute (get-first-object-attribute ?card ?arc))
    (while (neq ?attribute "")
     (if  (neq ?attribute ?name-attrib)
      then    (printout ?filename "\\item {\\bf " ?attribute ": } "
               (convert-to-latex (get-object-string-attribute ?card ?arc ?attribute)) crlf)
     )
     (bind ?attribute (get-next-object-attribute))
    )
    (printout ?filename "\\end{itemize}" crlf crlf)
    (bind ?counter (+ ?counter 1))
  )
)
      
(deffunction write-input-output (?card ?filename ?node ?node-image)
  (bind ?*to-list* (mv-append))
  (bind ?*from-list* (mv-append))
  (bind ?arc-image (get-first-node-image-arc ?card ?node-image))
  (while (neq ?arc-image -1)
    (bind ?to (get-arc-image-to ?card ?arc-image))    
    (bind ?from (get-arc-image-from ?card ?arc-image)) 
    (if (eq ?node-image ?to)
     then (bind ?*to-list* (mv-append ?*to-list* (get-object-from-image ?card ?arc-image)))
     else (if (eq ?node-image ?from)
           then (bind ?*from-list* (mv-append ?*from-list*
(get-object-from-image ?card ?arc-image)))
           else (printout t "-- Error: Arc doesn't link to node" crlf)
          )
    )
    (bind ?arc-image (get-next-node-image-arc))
  )
  (write-io ?card ?filename "Inputs" ?*to-list*)
  (write-io ?card ?filename "Outputs" ?*from-list*)
)

(deffunction write-nodes-and-arcs (?card ?filename)
  (set-status-message " Writing nodes and arcs...")
  (bind ?node-image (diagram-card-get-first-node-image ?card))
  (while (neq ?node-image -1)
    (bind ?node (get-object-from-image ?card ?node-image))
    (bind ?node-name (convert-to-latex (get-object-string-attribute ?card ?node ?*id-attribute*)))
    
    (if (neq ?node-name "") ;; i.e. if the "name" attribute exists and is defined
     then (printout ?filename "\\subsection{" ?node-name "}" crlf crlf)
     else (printout ?filename "\\subsection{Node}" crlf crlf)
    )
    (printout ?filename "\\begin{itemize}\\itemsep=0pt" crlf)
    (bind ?attribute (get-first-object-attribute ?card ?node))
    (while (neq ?attribute "")
     (if  (neq ?attribute ?*id-attribute*) 
      then    (printout ?filename "\\item {\\bf " (convert-to-latex ?attribute) ": } "
               (convert-to-latex (get-object-string-attribute ?card ?node ?attribute)) crlf)
     )
     (bind ?attribute (get-next-object-attribute))
    )
    ;;; Write references to the cards that this image is linked to
    (bind ?image-refs (mv-append))
    (bind ?item (diagram-image-get-item ?card ?node-image))
    (bind ?link (item-get-first-link ?card ?item))
    (while (neq ?link -1)
      (bind ?ref (link-get-card-to ?link))
      (if (neq ?ref ?card) then
       (bind ?image-refs (mv-append ?image-refs ?ref))
      )
      (bind ?link (item-get-next-link))
    )
    (if (neq 0 (length$ ?image-refs)) then
      (printout ?filename "\\item {\\bf Linked to:} ")
      (bind ?other-card (nth$ 1 ?image-refs))
      (while (> (length$ ?image-refs) 0)
        (printout ?filename "\\helpref{" (get-card-string-attribute ?other-card "title") "}{label" ?other-card "} ")
        (bind ?image-refs (rest$ ?image-refs))
        (bind ?other-card (nth$ 1 ?image-refs))
      )
    )

    
;    (write-input-output ?card ?filename ?node ?node-image)
    (printout ?filename "\\end{itemize}" crlf crlf) 
;    (write-arcs ?card ?filename ?*from-list*) 
    (bind ?node-image (diagram-card-get-next-node-image))
  )
  (bind ?*node-images-written* (mv-append))
  (set-status-message " Done writing nodes and arcs...")
)


(deffunction write-diagram (?card ?filename)
;  (bind ?printfile (get-card-string-attribute ?card "print-file"))
  (bind ?printfile (str-cat "diag" ?card))
;  (if (eq ?printfile "") 
;   then (bind ?printfile (get-text-from-user (str-cat "Enter print filename (no extension) for card "
;                                                      (get-card-string-attribute ?card "title")
;                                             )
;                         )
;        ))
  (if (neq ?printfile "") then
    (bind ?printfile (strip-extension ?printfile))
    (set-card-string-attribute ?card "print-file" ?printfile)

    ;;; Only save metafile under Windows 3
    (if (eq (get-platform) "Windows 3.1") then
      (set-status-message "Saving bitmap...")
;      (diagram-card-save-metafile ?card (str-cat ?printfile ".wmf") 4.0)
      (diagram-card-save-bitmap ?card (str-cat ?printfile ".bmp"))

    )
    
    (printout ?filename crlf crlf "\$\$" crlf)
    (printout ?filename "\\image{")
    (bind ?width (get-card-print-width ?card))
    (bind ?print-width (min 15 (* ?width 0.016))) 
                    ;; arbitrary max. width of 900 points - anything
                    ;; wider is scaled down ( 15 / 900 = 0.016). A default
                    ;; card is 600 points wide
    (printout ?filename ?print-width)
    (printout ?filename "cm;0cm}{")
    (printout ?filename ?printfile)
    (printout ?filename ".eps}" crlf "\$\$" crlf crlf)
  )
)

(deffunction write-linked-cards (?card ?filename)
  (bind ?linked-cards (mv-append))

  (bind ?item (card-get-first-item ?card))
  (while (neq ?item -1)
    (bind ?link (item-get-first-link ?card ?item))
   
    (while (neq ?link -1)
      (if (eq ?card (link-get-card-from ?link)) then
        (bind ?linked-cards (mv-append ?linked-cards (link-get-card-to ?link))))
      (bind ?link (item-get-next-link))
    )
    (bind ?item (card-get-next-item))
  )

 (bind ?other-card (nth$ 1 ?linked-cards))
 (while (> (length$ ?linked-cards) 0)
  (write-card ?other-card ?filename)
  (bind ?linked-cards (rest$ ?linked-cards))
  (bind ?other-card (nth$ 1 ?linked-cards))
 )
)

(deffunction write-links-to-other-cards (?card ?filename ?only-special-item)
  (bind ?linked-cards (mv-append))
  (bind ?special (card-get-special-item ?card))

  (bind ?item (card-get-first-item ?card))
  (while (neq ?item -1)
    (bind ?link (item-get-first-link ?card ?item))
    
    ;;; Sometimes you only want the special item links since you've
    ;;; covered the others (e.g. node images)
    (if (not (and (eq ?special ?item) ?only-special-item)) then
    
      (while (neq ?link -1)
        (if (eq ?card (link-get-card-from ?link)) then
          (bind ?linked-cards (mv-append ?linked-cards (link-get-card-to ?link))))
        (bind ?link (item-get-next-link))
      )
    )
    (bind ?item (card-get-next-item))
  )
  
  (printout ?filename "\\section{Links to other cards}" crlf crlf)
  
  (if (eq 0 (length$ ?linked-cards)) then
    (printout ?filename "None." crlf)
   else
    (printout ?filename "\\begin{itemize}\\itemsep=0pt" crlf)

   (bind ?other-card (nth$ 1 ?linked-cards))
   (while (> (length$ ?linked-cards) 0)
    (bind ?title (get-card-string-attribute ?other-card "title"))
    (printout ?filename "\\item \\helpref{" ?title "}{label" ?other-card "}" crlf)
    (bind ?linked-cards (rest$ ?linked-cards))
    (bind ?other-card (nth$ 1 ?linked-cards))
   )
   (printout ?filename "\\end{itemize}" crlf)
   
  )
)

(deffunction write-diagram-card (?card ?filename)
  (bind ?title (get-card-string-attribute ?card "title"))
  (printout ?filename "\\chapter")
  (printout ?filename "{" (convert-to-latex ?title) "}\\label{label" ?card "}" crlf)
  (write-diagram ?card ?filename)
  (printout ?filename "\\section{Nodes}" crlf crlf)
  (write-nodes-and-arcs ?card ?filename)
  (write-links-to-other-cards ?card ?filename TRUE)
  (write-linked-cards ?card ?filename)
)

(deffunction copy-text (?card ?newfile)
 (bind ?cardfile (hardy-path-search (card-get-string-attribute ?card "filename")))
 (if (neq ?cardfile "") then
  (if (not (open ?cardfile oldfile "r")) then
   (bind ?msg (str-cat "Could not open file " ?cardfile))
   (message-box ?msg)
   (return 0)
  )
  (bind ?line (readline oldfile))
  (while (and (neq ?line EOF) (neq ?line FALSE))
   (printout ?newfile ?line)
   (if (eq ?line "") then (printout ?newfile "\\par"))
   (printout ?newfile crlf)
   (bind ?line (readline oldfile))
  )
  (close oldfile)
 )
)

(deffunction write-text-card (?card ?filename)
  (bind ?title (get-card-string-attribute ?card "title"))
  (printout ?filename "\\chapter")
  (printout ?filename "{" (convert-to-latex ?title) "}\\label{label" ?card "}" crlf crlf)

  (copy-text ?card ?filename)
  (printout ?filename crlf)  
  (write-links-to-other-cards ?card ?filename FALSE)
  (write-linked-cards ?card ?filename)
)

(deffunction write-hypertext-card (?card ?filename)
  (bind ?title (get-card-string-attribute ?card "title"))
  (printout ?filename "\\chapter")
  (printout ?filename "{" (convert-to-latex ?title) "}\\label{label" ?card "}" crlf crlf)

  (printout ?filename "Hypertext card output is currently unimplemented." crlf)
  (write-links-to-other-cards ?card ?filename FALSE)
  (write-linked-cards ?card ?filename)
)

(deffunction write-card (?card ?filename)
  (yield)
  (bind ?title (get-card-string-attribute ?card "title"))

 (bind ?card-type (card-get-string-attribute ?card "type"))

  (switch ?card-type
   (case "Diagram card" then (write-diagram-card ?card ?filename))
   (case "Diagram expansion" then (write-diagram-card ?card ?filename))
   (case "Text card" then (write-text-card ?card ?filename))
   (case "Hypertext card" then (write-hypertext-card ?card ?filename))
  )
)

(deffunction write-all (?card-id ?file)
  (begin-status-dialog)

  (open ?file File "w")
  (printout File "\\documentstyle[a4,12pt,texhelp]{report}" crlf) 
  (printout File "\\input psbox.tex" crlf  crlf)
  (printout File "\\begin{document}" crlf)
  (printout File "\\title{") 
  (printout File "HARDY generated document")
  (printout File "}" crlf) 
  (printout File "\\author{HARDY}" crlf)
  (printout File "\\date{\\today}" crlf)
  (printout File "\\maketitle" crlf)
  (printout File "\\tableofcontents" crlf  crlf)
  (bind ?*expansion-cards* (mv-append))
  (write-card ?card-id File)
  (printout File "\\end{document}" crlf)
  (close File)

  (end-status-dialog)
)

;;; Sample event handler for writing PostScript report.
;;; See ddeword.clp for writing an RTF report.
(deffunction test-event-handler (?card-id ?option)
;  (if (eq ?option "Write LaTeX file") then
;      (bind ?file (get-text-from-user "Please give the name of the LaTeX file
;to be written" ""))
;   (print-hierarchy-to-files ?card-id)
;   (write-all ?card-id ?file)
;  )  
;)
