;;; convert.clp
;;; Hardy2HTML main conversion routines

;;; Finds a card HTML filename for the card id
(defmessage-handler MAIN::HtmlWriter find-card-file (?card-id)
 (return (str-cat "card-" ?card-id ".html")))

;;; Write an HTML list of links to other cards from the special item
;;; (for links not from particular items on the card such as images
;;; or blocks).
;;; Returns the number of links found.

(defmessage-handler MAIN::HtmlWriter write-special-links (?card ?file ?is-frame)

 (if (eq ?is-frame TRUE) then
   (printout ?file "<CENTER><H3>Linked information for " (card-get-string-attribute ?card "title") "</H3></CENTER>" crlf)
  else
   (printout ?file "<CENTER><H3>Linked information</H3></CENTER>" crlf)
 )
 
 (printout ?file "<UL>" crlf)
 (bind ?count 0)
 (bind ?itemid (card-get-special-item ?card))

 (bind ?linkid (item-get-first-link ?card ?itemid))
 (while (neq ?linkid -1)
   (bind ?othercard (link-get-card-to ?linkid))
   (if (neq ?othercard ?card) then
    (if (eq ?is-frame TRUE) then
      (printout ?file (str-cat "<LI><a href=card-" ?othercard "-frame.html TARGET=topwindow>"))
     else
      (printout ?file (str-cat "<LI><a href=card-" ?othercard ".html>")))
    (printout ?file (card-get-string-attribute ?othercard "title"))
    (printout ?file (str-cat "</a>") crlf)
    (bind ?count (+ ?count 1))
   )
   (bind ?linkid (item-get-next-link))
 )
 (printout ?file "</UL>" crlf)
 
 (if (eq ?count 0) then
   (printout ?file "No links.<P>")
 )
 
 (if (eq ?is-frame TRUE) then
   (printout ?file "<HR><I><FONT SIZE=2>Back to <A HREF=card-" ?card "-frame.html TARGET=topwindow>" (card-get-string-attribute ?card "title") "</FONT></A></I>" crlf)
 )
 
 (return ?count)
)

;;; Convert GIF files from BMPs by invoking a DOS utility.

(defmessage-handler MAIN::HtmlWriter convert-files ()
 (bind ?filelist (str-cat ?self:output-path "/gifs.lst"))
 (bind ?command (str-cat ?self:gif-converter " -O -G @" ?filelist))
 (printout t "Command is: " ?command crlf)

 (open ?filelist gifs "w")

 (bind ?files ?self:bitmap-file-list)

 (bind ?name (nth$ 1 ?files))
 (while (> (length$ ?files) 0)
  (printout gifs ?name ".bmp" crlf)
  (bind ?files (rest$ ?files))
  (bind ?name (nth$ 1 ?files))
 )
 (close gifs)
 (printout t ?command crlf)
 (execute ?command 1)
)
  
;;; Custom menu handler (toplevel Hardy window).
(deffunction trans-custom-menu (?menuitem)
  (if (eq ?menuitem "Translate to &HTML") then (send ?*html-writer* translate-index)
  )
)

;;; Translate a card.
(defmessage-handler MAIN::HtmlWriter translate-card (?card)
 (bind ?card-type (card-get-string-attribute ?card "type"))
 (send ?self tell-processing (card-get-string-attribute ?card "title"))
 (printout t "Processing card " (card-get-string-attribute ?card "title") crlf)
 (switch ?card-type
  (case "Diagram card" then (send ?self translate-diagram-card ?card))
  (case "Diagram expansion" then (send ?self translate-diagram-card ?card))
  (case "Text card" then (send ?self translate-text-card ?card))
  (case "Hypertext card" then (send ?self translate-hypertext-card ?card))
 )
 (card-quit ?card)
)

;;; Translate a card and then its children, adding nested lists to
;;; the contents page to reproduce the Hardy card hierarchy.
(defmessage-handler MAIN::HtmlWriter translate-card-recursive (?card ?level)
  (bind ?*recursion-level* ?level)
  
  (if (send ?self is-card-processed ?card) then (return FALSE))

  (send ?self translate-card ?card)
  
  (bind ?children (create$))
  
  ;;; Now collect a list of children of this card, if any
  (bind ?item-id (card-get-first-item ?card))
  (while (neq ?item-id -1)
    (bind ?link-id (item-get-first-link ?card ?item-id))
    (while (neq ?link-id -1)
      (bind ?card-to (link-get-card-to ?link-id))
      (if (neq ?card-to ?card) then
        (bind ?children (mv-append ?children ?card-to))
      )
       
      (bind ?link-id (item-get-next-link))
    )
    (bind ?item-id (card-get-next-item))
  )
  
  (if (> (length$ ?children) 0) then
    (printout toplevelstream "<UL>" crlf)
    (printout toplevelframestream "<UL>" crlf)
    
    (while (> (length$ ?children) 0)
      (bind ?child (nth$ 1 ?children))
     
      (send ?self translate-card-recursive ?child (+ ?level 1))
     
      (bind ?children (rest$ ?children)))

    (printout toplevelstream "</UL>" crlf)
    (printout toplevelframestream "</UL>" crlf)
  )
)

;;; Translate index (i.e. all cards)
(defmessage-handler MAIN::HtmlWriter translate-index ()
;  (unwatch all)
  (send ?self reset)
  
  ;;; Read preferences from .ini file
  (send ?self read-preferences)

  ;;; Show the preferences dialog
  (bind ?ok (send ?self show-preferences-dialog))
  (if (eq ?ok FALSE) then (return FALSE))
  
  ;;; Write preferences to .ini file
  (send ?self write-preferences)

  ;;; Check if the project directory exists
  (if (eq 0 (dir-exists ?self:output-path)) then
   (if (eq 0 (mkdir ?self:output-path)) then
    (message-box (str-cat "Sorry, could not create directory " ?self:output-path))
    (return FALSE))
  )

  ;;; Copy standard documents (help.htm etc.) to the destination directory.
  (if (eq FALSE (send ?self copy-standard-documents)) then
    (return FALSE))

  (send ?self show-processing-dialog)
  
  ;;; First output the frame/noframe top-level.html file
  
  (open (str-cat ?self:output-path "/" ?self:top-level-file) toplevelstream "w")
  (send ?self output-toplevel-header toplevelstream)

  ;;; Output frameset stuff. We have:
  ;;;
  ;;;  --------------------------------------
  ;;; |                                      |
  ;;; |            topwindow                 |
  ;;; |        (for main card info)          |
  ;;; |                                      |
  ;;; |--------------------------------------|
  ;;; |           |                          |
  ;;; | contents  |      infowindow          |
  ;;; |           |(for details e.g. attribs)|
  ;;;  --------------------------------------
  ;;;
  
  (bind ?top-card (hardy-get-top-card))

  (printout toplevelstream "<FRAMESET ROWS=\"50%,50%\">" crlf)
  (printout toplevelstream "<FRAME SRC=\"card-" ?top-card "-frame.html\" NAME=topwindow>" crlf)
  (printout toplevelstream "  <FRAMESET COLS=\"40%,60%\">" crlf)
  (printout toplevelstream "  <FRAME SRC=\"top-level-frame.html\" NAME=contents>" crlf)
  (printout toplevelstream "  <FRAME SRC=\"help.html\" NAME=infowindow>" crlf)
  (printout toplevelstream "  </FRAMESET>" crlf)
  (printout toplevelstream "</FRAMESET>" crlf)
  
  ;;; That's the frame-enabled stuff. Now for HTML to allow non-frame
  ;;; browsers to cope.
  
  (printout toplevelstream "<NOFRAMES>" crlf)
  (send ?self output-toplevel-body toplevelstream "bg.gif")
  
;  (printout toplevelstream "<H2>Contents</H2>" crlf)
  (printout toplevelstream "<IMG ALT=\"Contents\" SRC=contents.gif><P>" crlf)
;  (printout toplevelstream "<UL>")
  (printout toplevelstream crlf)
  
  ;;; Open a file for the frame contents: similar to top-level.html
  ;;; but just the list of contents.
  (open (str-cat ?self:output-path "/top-level-frame.html") toplevelframestream "w")
  (send ?self output-toplevel-header toplevelframestream)
  (send ?self output-toplevel-body toplevelframestream "bg-2.gif")
;  (printout toplevelframestream "<H2>Contents</H2>" crlf)
  (printout toplevelframestream "<IMG ALT=\"Contents\" SRC=contents.gif><P>" crlf)
;  (printout toplevelframestream "<UL>")
  (printout toplevelframestream crlf)

  ;;; Processing starts from the top card, in order to generate
  ;;; a hierarchy corresponding to the Hardy hierarchy.
  (if (neq ?top-card -1) then
   (send ?self translate-card-recursive ?top-card 0))
   
  ;;; Now process any cards that haven't been done recursively, e.g.
  ;;; orphan cards
  (bind ?card (hardy-get-first-card))
  (while (neq ?card -1)
    (if (not (send ?self is-card-processed ?card)) then
      (send ?self translate-card ?card))
     
    (bind ?card (hardy-get-next-card))
  )

;  (printout toplevelstream "</UL>")
  (printout toplevelstream crlf)
;  (printout toplevelframestream "</UL>")
  (printout toplevelframestream crlf)

  ;;; Output footer for contents page.
  (send ?self output-toplevel-footer toplevelstream "</NOFRAMES>")
  (send ?self output-toplevel-footer toplevelframestream "")

  (close toplevelstream)
  (close toplevelframestream)
  
  ;; Convert the BMP files to GIFs.
  (send ?self tell-processing "Converting BMPs to GIFs")

  (if (> (length$ ?self:bitmap-file-list) 0) then
   (send ?self convert-files)
  )
  (message-box (str-cat "Translated Hardy index." ?*nl* "Press OK when GIFs have been translated." ?*nl*
     "This will delete the BMP files."))
  
  (send ?self hide-processing-dialog)

  ;;; Delete all the BMP files
  (bind ?files ?self:bitmap-file-list)
  (bind ?name (nth$ 1 ?files))
  (while (> (length$ ?files) 0)
   (remove (str-cat ?name ".bmp"))
   (bind ?files (rest$ ?files))
   (bind ?name (nth$ 1 ?files))
  )
  ;;; Remove gifs.lst
  (remove (str-cat ?self:output-path "/gifs.lst"))
  (return TRUE)
)

;;; Copy files which are the same for each report
(defmessage-handler MAIN::HtmlWriter copy-standard-documents ()
 (if (eq 0 (dir-exists ?self:standard-path)) then
   (message-box (str-cat "Standard report file path " ?self:standard-path
" does not exist.
Please set this to the location where help.htm and other files can be found."))
   (return FALSE)
 )

 (bind ?help-file (str-cat ?self:standard-path "/help.htm"))
 (bind ?noexp-file (str-cat ?self:standard-path "/noexp.htm"))
 (bind ?contentsgif-file (str-cat ?self:standard-path "/contents.gif"))
 (bind ?background-file (str-cat ?self:standard-path "/bg.gif"))
 (bind ?background-file1 (str-cat ?self:standard-path "/bg-1.gif"))
 (bind ?background-file2 (str-cat ?self:standard-path "/bg-2.gif"))
 (bind ?background-file3 (str-cat ?self:standard-path "/bg-3.gif"))

 (bind ?help-file-target (str-cat ?self:output-path "/help.html"))
 (bind ?noexp-file-target (str-cat ?self:output-path "/noexp.html"))
 (bind ?contentsgif-target (str-cat ?self:output-path "/contents.gif"))
 (bind ?background-target (str-cat ?self:output-path "/bg.gif"))
 (bind ?background-target1 (str-cat ?self:output-path "/bg-1.gif"))
 (bind ?background-target2 (str-cat ?self:output-path "/bg-2.gif"))
 (bind ?background-target3 (str-cat ?self:output-path "/bg-3.gif"))

 (if (or (eq 0 (file-exists ?help-file))
         (eq 0 (file-exists ?noexp-file))
         (eq 0 (file-exists ?contentsgif-file))
         (eq 0 (file-exists ?background-file))) then
   (message-box (str-cat "Could not find one or more standard report file."))
   (return FALSE)
 )
 
 (if (eq 0 (copy-file ?help-file ?help-file-target)) then
   (message-box (str-cat "Could not copy " ?help-file)))

 (if (eq 0 (copy-file ?noexp-file ?noexp-file-target)) then
   (message-box (str-cat "Could not copy " ?noexp-file)))

 (if (eq 0 (copy-file ?contentsgif-file ?contentsgif-target)) then
   (message-box (str-cat "Could not copy " ?contentsgif-file)))

 (if (eq 0 (copy-file ?background-file ?background-target)) then
   (message-box (str-cat "Could not copy " ?background-file)))

 (if (eq 0 (copy-file ?background-file1 ?background-target1)) then
   (message-box (str-cat "Could not copy " ?background-file1)))

 (if (eq 0 (copy-file ?background-file2 ?background-target2)) then
   (message-box (str-cat "Could not copy " ?background-file2)))

 (if (eq 0 (copy-file ?background-file3 ?background-target3)) then
   (message-box (str-cat "Could not copy " ?background-file3)))

 (return TRUE)
)

;;; Output contents page header.
(defmessage-handler MAIN::HtmlWriter output-toplevel-header (?stream)
 
 (printout ?stream "<HTML>" crlf)
 (printout ?stream "<HEAD>" crlf "<TITLE>" ?self:title "</TITLE>" crlf)
 (printout ?stream "</HEAD>" crlf crlf)
)

(defmessage-handler MAIN::HtmlWriter output-toplevel-body (?stream ?background)
 (send ?self output-body-tag ?stream ?background)
 (printout ?stream "<CENTER>" crlf)
 (printout ?stream "<H2>" ?self:title "</H2>" crlf)
 (printout ?stream "<I><B>" ?self:author "</B></I><BR>" crlf)
 (printout ?stream "<B>" (now) "</B><BR>" crlf)
 (printout ?stream "</CENTER>" crlf)
 (printout ?stream "<HR>" crlf)

)

;;; Output contents page footer.
(defmessage-handler MAIN::HtmlWriter output-toplevel-footer (?stream ?noframes)

 (printout ?stream "<HR><B><A HREF=\"help.html\" TARGET=infowindow>Help</A></B>" crlf)
 (printout ?stream "<HR><I><FONT SIZE=2><A HREF=\"http://www.aiai.ed.ac.uk/~hardy/\" TARGET=\"_top\">Hardy</A> to HTML converter, (c)
  <A HREF=\"http://www.aiai.ed.ac.uk/\" TARGET=\"_top\">AIAI</A>, University of Edinburgh, 1996</FONT></I>" crlf)
 (printout ?stream "</BODY>" crlf)
 (printout ?stream ?noframes crlf)
 (printout ?stream "</HTML>" crlf)
)

;;; Output a pointer to a card page from the contents page.
(defmessage-handler MAIN::HtmlWriter output-card-contents-entry (?stream ?title ?card-url)
;  (if (> ?*recursion-level* 0) then (printout ?stream "<LI>"))
  (printout ?stream "<LI>")
  (printout ?stream "<A HREF=\"" ?card-url "\" TARGET=topwindow>" ?title "</A>" crlf)
)

;;; Output a card footer.
(defmessage-handler MAIN::HtmlWriter output-card-footer (?stream ?card)

 (printout ?stream "<P><HR><B><A HREF="top-level.html" TARGET=_top>Contents</A> <A HREF=\"help.html\" TARGET=infowindow>Help</A></B>" crlf)
 (printout ?stream "<HR><I><FONT SIZE=2><A HREF=\"http://www.aiai.ed.ac.uk/~hardy/\" TARGET=_top>Hardy</A> to HTML converter, (c)
  <A HREF=\"http://www.aiai.ed.ac.uk/\" TARGET=_top>AIAI</A>, University of Edinburgh, 1996</FONT></I>" crlf)
 (printout ?stream "</BODY>" crlf)
 (printout ?stream "</HTML>" crlf)
)

;;; Output body tag, with background/text colour.
(defmessage-handler MAIN::HtmlWriter output-body-tag (?stream ?background)

 ;;; White background
; (printout ?stream "<BODY BGCOLOR=#FFFFFF TEXT=#000000>" crlf)

 ;;; Light grey background
 (printout ?stream "<BODY BACKGROUND=" ?background " BGCOLOR=#DDDDDD TEXT=#000000>" crlf)
)

;;; Read preferences from hardyhtm.ini.
(defmessage-handler MAIN::HtmlWriter read-preferences ()
  (bind ?self:title (get-resource "HardyHtml" "title" "hardyhtm.ini"))
  (if (eq "" ?self:title) then (bind ?self:title "Unnamed Report"))

  (bind ?self:author (get-resource "HardyHtml" "author" "hardyhtm.ini"))
  (if (eq "" ?self:author) then (bind ?self:author "Unnamed author"))

  (bind ?self:output-path (get-resource "HardyHtml" "outputPath" "hardyhtm.ini"))
  (if (eq "" ?self:output-path) then (bind ?self:output-path "c:/html/test"))

  (bind ?self:name-attribute (get-resource "HardyHtml" "nameAttribute" "hardyhtm.ini"))
  (if (eq "" ?self:name-attribute) then (bind ?self:name-attribute "name"))

  (bind ?self:standard-path (get-resource "HardyHtml" "standardPath" "hardyhtm.ini"))
  (if (eq "" ?self:standard-path) then (bind ?self:standard-path "c:/hrdyhtml"))
)

;;; Write preferences to hardyhtm.ini.
(defmessage-handler MAIN::HtmlWriter write-preferences ()
  (write-resource "HardyHtml" "title" ?self:title "hardyhtm.ini")
  (write-resource "HardyHtml" "author" ?self:author "hardyhtm.ini")
  (write-resource "HardyHtml" "outputPath" ?self:output-path "hardyhtm.ini")
  (write-resource "HardyHtml" "standardPath" ?self:standard-path "hardyhtm.ini")
  (write-resource "HardyHtml" "nameAttribute" ?self:name-attribute "hardyhtm.ini")
)
