;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; LaTeX document-writing CLIPS code.
;;; Call (write-all ?card-id ?filename) from a suitable event handler.
;;; Originally by JKK; modified by JACS (10/7/94) to support Tex2RTF
;;; and to give user feedback under Windows (not possible under X)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; BUGS: 
;;  1/ LaTeX special characters (_, &, $, %) are not given the appropriate
;;     escape character (\).
;;     CURED, JACS 16/7/94
;;  2/ The first attribute (apart from 'type') of nodes & arcs is assumed to
;;     be the name or label, and is used to identify the node/arc.
;;  3/ If the card is wider than 900 points, the diagram is scaled down. This
;;     is an arbitrary limit. (A default card is 600 points wide).
;;  4/ It's possible to create \subsubsubsections (or even deeper), which
;;     LaTeX does not support. 
;;
;; Other files needed: utils1.clp


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Global variables
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defglobal 
  ?*leftmost-coordinate* = 10000
  ?*leftmost-node-image* = FALSE
  ?*node-images-written* = (mv-append)
  ?*to-list* = (mv-append)
  ?*from-list* = (mv-append)
  ?*expansion-cards* = (mv-append)
  ?*status-dialog* = 0
  ?*status-message* = 0
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Forward declarations
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(deffunction write-card (?one ?two ?three ?four))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Utility functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(deffunction find-arc-from-id (?card ?id)
  (bind ?arc (get-first-card-arc ?card))
  (while (neq ?arc -1)
    (if (eq ?id ?arc)
     then (return ?arc)
     else (bind ?arc (get-next-card-arc))
    )
  )
  (return FALSE) ;; arc not found
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Conversion from normal text to Latex-compatible text (e.g. ampersand conversion)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(deffunction convert-to-latex (?text)
 (bind ?out-string "")
 (bind ?len (length$ ?text))
 (bind ?i 1)
 (while (<= ?i ?len) do
  (bind ?char (sub-string ?i ?i ?text))
  (switch ?char
    (case "&" then
     (bind ?out-string (str-cat ?out-string "\\&"))
    )
    (case "%" then
     (bind ?out-string (str-cat ?out-string "\\%"))
    )
    (case "_" then
     (bind ?out-string (str-cat ?out-string "\\_"))
    )
    (case "$" then
     (bind ?out-string (str-cat ?out-string "\\$"))
    )
    (default 
     (bind ?out-string (str-cat ?out-string ?char))
    )
  )
  (bind ?i (+ ?i 1))
 )
 (return ?out-string)
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; User feedback (Windows only)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(deffunction begin-status-dialog ()
 (if (eq (get-platform) "Windows 3.1") then
  (bind ?*status-dialog* (dialog-box-create 0 "Creating Report" 0))
  (bind ?*status-message* (message-create ?*status-dialog* "Creating report, please wait..."))
  (window-fit ?*status-dialog*)
  (window-show ?*status-dialog* 1)
  (yield)
 )
)

(deffunction end-status-dialog ()
 (if (eq (get-platform) "Windows 3.1") then
  (window-show ?*status-dialog* 0)
  (window-delete ?*status-dialog*)
  (bind ?*status-dialog* 0)
  (yield)
 )
)

(deffunction set-status-message (?text)
 (if (eq (get-platform) "Windows 3.1") then
  (panel-item-set-label ?*status-message* ?text)
  (yield)
 )
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Main functions - the top level functions appear last
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Find-leftmost-etc works by checking coordinates of node IMAGES, since these
;; are specific to individual expansion cards, whereas nodes are not

(deffunction find-leftmost-node-image-not-written (?card) 
  (bind ?*leftmost-coordinate* 10000)
  (bind ?*leftmost-node-image* FALSE)
  (bind ?node-image (get-first-card-node-image ?card))
  (while (neq ?node-image -1)
    (bind ?image-x (get-image-x ?card ?node-image))
    (if (and (or (< ?image-x ?*leftmost-coordinate*) 
  ;; If the image is to the left of the leftmost image found so far ...
                 (and (eq ?image-x ?*leftmost-coordinate*)
                      (bind ?image-y (get-image-y ?card ?node-image))
                      (< ?image-y (get-image-y ?card ?*leftmost-node-image*))
                 )
             )
  ;; ... or it is immediately above the leftmost image found so far ...
             (not (member ?node-image ?*node-images-written*))
  ;; ... and it hasn't already been written out ...
        )
     then (bind ?*leftmost-coordinate* ?image-x)
          (bind ?*leftmost-node-image* ?node-image)
    )
  ;; ... then instantiate it to be the leftmost so far.
    (bind ?node-image (get-next-card-node-image))
  )
  (return ?*leftmost-node-image*)          
)


(deffunction write-io (?card ?filename ?attribute ?list)
  (yield)
  (printout ?filename "\\item {\\bf " ?attribute ":} ")
  (bind ?counter 1)
  (bind ?length (length ?list))
  (while (<= ?counter ?length)  
    (bind ?arc (nth ?counter ?list))
    (bind ?name-attrib (get-first-object-attribute ?card ?arc))
    (if (eq ?name-attrib "type") 
     then (bind ?name-attrib (get-next-object-attribute))
    )    
    (bind ?arc-name (convert-to-latex (get-object-string-attribute ?card ?arc ?name-attrib)))
    (if (> ?counter 1) then (printout ?filename ", "))
    (if (neq ?arc-name "") ;; i.e. if an attribute exists and is defined
     then (printout ?filename (convert-to-latex ?arc-name))
     else (printout ?filename (get-object-string-attribute ?card ?arc "type"))
    )
    (bind ?counter (+ ?counter 1))
  )
  (printout ?filename crlf)
)

(deffunction write-arcs (?card ?filename ?list)
  (yield)
  (bind ?counter 1)
  (bind ?length (length ?list))
  (while (<= ?counter ?length)  
    (bind ?arc (nth ?counter ?list))
    (bind ?name-attrib (get-first-object-attribute ?card ?arc))
    (if (eq ?name-attrib "type") 
     then (bind ?name-attrib (get-next-object-attribute))
    )    
    (bind ?arc-name (convert-to-latex (get-object-string-attribute ?card ?arc ?name-attrib)))
    (if (neq ?arc-name "") ;; i.e. if the "name" attribute exists and is defined
     then (printout ?filename "\\item {\\em " (convert-to-latex ?arc-name) "}" crlf)
     else (printout ?filename "\\item {\\em " Arc "}" crlf)
    )
    (printout ?filename "\\begin{itemize}" crlf)
    (bind ?attribute (get-first-object-attribute ?card ?arc))
    (while (neq ?attribute "")
     (if  (neq ?attribute ?name-attrib)
      then    (printout ?filename "\\item {\\bf " ?attribute ": } "
               (convert-to-latex (get-object-string-attribute ?card ?arc ?attribute)) crlf)
     )
     (bind ?attribute (get-next-object-attribute))
    )
    (printout ?filename "\\end{itemize}" crlf crlf)
    (bind ?counter (+ ?counter 1))
  )
)
      
(deffunction write-input-output (?card ?filename ?node ?node-image)
  (bind ?*to-list* (mv-append))
  (bind ?*from-list* (mv-append))
  (bind ?arc-image (get-first-node-image-arc ?card ?node-image))
  (while (neq ?arc-image -1)
    (bind ?to (get-arc-image-to ?card ?arc-image))    
    (bind ?from (get-arc-image-from ?card ?arc-image)) 
    (if (eq ?node-image ?to)
     then (bind ?*to-list* (mv-append ?*to-list* (get-object-from-image ?card ?arc-image)))
     else (if (eq ?node-image ?from)
           then (bind ?*from-list* (mv-append ?*from-list*
(get-object-from-image ?card ?arc-image)))
           else (printout t "-- Error: Arc doesn't link to node" crlf)
          )
    )
    (bind ?arc-image (get-next-node-image-arc))
  )
  (write-io ?card ?filename "Inputs" ?*to-list*)
  (write-io ?card ?filename "Outputs" ?*from-list*)
)

;; This function makes sure expansion cards are written out in the same order
;; as the nodes in the diagram

(deffunction note-expansion-cards (?card ?node-image)  
  (bind ?expansion (get-first-image-expansion ?card ?node-image))
  (while (neq ?expansion -1)
    (bind ?*expansion-cards* (mv-append ?*expansion-cards* ?expansion))
    (bind ?expansion (get-next-image-expansion))
  )
)

(deffunction write-nodes-and-arcs-from-left (?card ?filename)
  (set-status-message "Writing nodes and arcs...")
  (bind ?node-image (find-leftmost-node-image-not-written ?card))
  (while (neq ?node-image FALSE)
    (bind ?*node-images-written* (mv-append ?node-image ?*node-images-written*))
    (bind ?node (get-object-from-image ?card ?node-image))
    (note-expansion-cards ?card ?node-image)
    (bind ?node-name (convert-to-latex (get-object-string-attribute ?card ?node "name")))
    (if (neq ?node-name "") ;; i.e. if the "name" attribute exists and is defined
     then (printout ?filename "\\item " ?node-name crlf)
     else (printout ?filename "\\item " Node crlf)
    )
    (printout ?filename "\\begin{itemize}" crlf)
    (bind ?attribute (get-first-object-attribute ?card ?node))
    (while (neq ?attribute "")
     (if  (neq ?attribute "name") 
      then    (printout ?filename "\\item {\\bf " (convert-to-latex ?attribute) ": } "
               (convert-to-latex (get-object-string-attribute ?card ?node ?attribute)) crlf)
     )
     (bind ?attribute (get-next-object-attribute))
    )
    (write-input-output ?card ?filename ?node ?node-image)
    (printout ?filename "\\end{itemize}" crlf crlf) 
    (write-arcs ?card ?filename ?*from-list*) 
    (bind ?node-image (find-leftmost-node-image-not-written ?card))
  )
  (bind ?*node-images-written* (mv-append))
)  


(deffunction write-diagram (?card ?filename)
  (bind ?printfile (get-card-string-attribute ?card "print-file"))
  (if (eq ?printfile "") 
   then (bind ?printfile (get-text-from-user (str-cat "Enter print filename (no extension) for card "
                                                      (get-card-string-attribute ?card "title")
                                             )
                         )
        ))
  (if (neq ?printfile "") then
    (bind ?printfile (strip-extension ?printfile))
    (set-card-string-attribute ?card "print-file" ?printfile)

    ;;; Only save metafile under Windows 3
    (if (eq (get-platform) "Windows 3.1") then
      (set-status-message "Saving metafile...")
      (diagram-card-save-metafile ?card (str-cat ?printfile ".wmf") 4.0))

    (printout ?filename crlf crlf "\$\$" crlf)
    (printout ?filename "\\image{")
    (bind ?width (get-card-print-width ?card))
    (bind ?print-width (min 15 (* ?width 0.016))) 
                    ;; arbitrary max. width of 900 points - anything
                    ;; wider is scaled down ( 15 / 900 = 0.016). A default
                    ;; card is 600 points wide
    (printout ?filename ?print-width)
    (printout ?filename "cm;0cm}{")
    (printout ?filename ?printfile)
    (printout ?filename "}" crlf "\$\$" crlf crlf)
  )
)

(deffunction write-linked-cards (?card ?filename ?depth ?n-remaining-items)
  (bind ?length (length ?*expansion-cards*))
  (while (> ?length ?n-remaining-items)
    (bind ?exp-card (nth (+ 1 ?n-remaining-items) ?*expansion-cards*))
    (bind ?*expansion-cards* (mv-delete (+ 1 ?n-remaining-items) ?*expansion-cards*))
    (bind ?length (- ?length 1))
    (write-card ?exp-card ?filename (+ ?depth 1) (max ?length 0))
  )
)

(deffunction print-subsections (?filename ?depth)
  (if (> ?depth 1)
   then (printout ?filename "sub")
        (print-subsections ?filename (- ?depth 1))
  )
)

(deffunction write-card (?card ?filename ?depth ?n-items)
  (yield)
  (bind ?title (get-card-string-attribute ?card "title"))
  (printout ?filename "\\")
  (print-subsections ?filename ?depth)
  (printout ?filename "section{" (convert-to-latex ?title) "}" crlf crlf)
  (write-diagram ?card ?filename)
  (printout ?filename "\\begin{itemize}" crlf)
  (write-nodes-and-arcs-from-left ?card ?filename)
  (printout ?filename "\\end{itemize}" crlf crlf)
  (write-linked-cards ?card ?filename ?depth ?n-items)
)

(deffunction write-all (?card-id ?file)
  (begin-status-dialog)

  (open ?file File "w")
  (printout File "\\documentstyle[a4,12pt,texhelp]{article}" crlf) 
  (printout File "\\input psbox.tex" crlf  crlf)
  (printout File "\\begin{document}" crlf)
  (printout File "\\title{") 
  (printout File "HARDY generated document")
  (printout File "}" crlf) 
  (printout File "\\author{HARDY}" crlf)
  (printout File "\\date{\\today}" crlf)
  (printout File "\\maketitle" crlf)
  (printout File "\\tableofcontents" crlf  crlf)
  (bind ?*expansion-cards* (mv-append))
  (write-card ?card-id File 1 0)
  (printout File "\\end{document}" crlf)
  (close File)

  (end-status-dialog)
)

;;; Sample event handler for writing PostScript report.
;;; See ddeword.clp for writing an RTF report.
(deffunction test-event-handler (?card-id ?option)
;  (if (eq ?option "Write LaTeX file") then
;      (bind ?file (get-text-from-user "Please give the name of the LaTeX file
;to be written" ""))
;   (print-hierarchy-to-files ?card-id)
;   (write-all ?card-id ?file)
;  )  
;)

; We are interested in CustomMenu events for KADS Inference diagrams
; (register-event-handler CustomMenu "KADS Inference" test-event-handler)

