;;; diagram.clp
;;; Converts diagram cards to HTML

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Forward declarations
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(deffunction write-nodes-and-arcs-from-left (?card ?filename))
(deffunction trans-diagram-output-link (?file ?card ?node-image ?start))

;;;
;;; Translate a diagram card
;;;

(deffunction trans-translate-diagram-card (?card)
 (trans-add-card ?card)
 (bind ?filename (str-cat ?*trans-output-path* "/" (trans-find-card-file ?card)))

 ;;; Open the HTML file
 (open ?filename tempfile "w")

 ;;; Output the title
 (printout tempfile "<TITLE>" (card-get-string-attribute ?card "title") "</TITLE>" crlf)
 (printout tempfile "<H1>" (card-get-string-attribute ?card "title") "</H1>" crlf crlf)

 ;;; Save a bitmap file and remember the name
 (bind ?bitmap-file-root (str-cat ?*trans-output-path* "/" ?*trans-name-root* ?*trans-file-id*))
 (bind ?bitmap-file-actual-root (str-cat ?*trans-actual-path* "/" ?*trans-name-root* ?*trans-file-id*))
 (bind ?*trans-file-id* (+ ?*trans-file-id* 1))
 (bind ?bitmap-filename (str-cat ?bitmap-file-root ".bmp"))
 (bind ?*trans-bitmap-file-list* (mv-append ?*trans-bitmap-file-list* ?bitmap-file-root))

 (diagram-card-save-bitmap ?card ?bitmap-filename)

 (bind ?inline-gif-filename (str-cat ?bitmap-file-actual-root ".gif"))
 (bind ?big-gif-filename (str-cat ?bitmap-file-actual-root "b.gif"))

 ;;; Output inline GIF
 (printout tempfile "<a href=\"" ?big-gif-filename "\"><img align=top src=\"" ?inline-gif-filename "\"></a><P>" crlf crlf)

 ;;; Now append any links from the Special Item
 (printout tempfile "<HR>" crlf)
 (if (> (trans-write-special-links ?card tempfile) 0) then
   (printout tempfile "<HR>" crlf)
 )

 ;;; Output nodes and arcs, with possible links to other cards
 (printout tempfile "<UL>" crlf)
 (write-nodes-and-arcs-from-left ?card tempfile)
 (printout tempfile "</UL>" crlf crlf)

 (close tempfile)
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Global variables
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defglobal 
  ?*leftmost-coordinate* = 10000
  ?*leftmost-node-image* = FALSE
  ?*node-images-written* = (mv-append)
  ?*to-list* = (mv-append)
  ?*from-list* = (mv-append)
  ?*expansion-cards* = (mv-append)
  ?*status-dialog* = 0
  ?*status-message* = 0
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Utility functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(deffunction find-arc-from-id (?card ?id)
  (bind ?arc (get-first-card-arc ?card))
  (while (neq ?arc -1)
    (if (eq ?id ?arc)
     then (return ?arc)
     else (bind ?arc (get-next-card-arc))
    )
  )
  (return FALSE) ;; arc not found
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Conversion from normal text to HTML-compatible text (e.g. ampersand conversion)
;;; No conversions as yet.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(deffunction convert-to-html (?text)
 (return ?text)

 ;;; This ignored pro tem
 (bind ?out-string "")
 (bind ?len (length$ ?text))
 (bind ?i 1)
 (while (<= ?i ?len) do
  (bind ?char (sub-string ?i ?i ?text))
  (switch ?char
    (case "&" then
     (bind ?out-string (str-cat ?out-string "&"))
    )
    (case "%" then
     (bind ?out-string (str-cat ?out-string "%"))
    )
    (case "_" then
     (bind ?out-string (str-cat ?out-string "_"))
    )
    (case "$" then
     (bind ?out-string (str-cat ?out-string "$"))
    )
    (default 
     (bind ?out-string (str-cat ?out-string ?char))
    )
  )
  (bind ?i (+ ?i 1))
 )
 (return ?out-string)
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Main functions - the top level functions appear last
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Find-leftmost-etc works by checking coordinates of node IMAGES, since these
;; are specific to individual expansion cards, whereas nodes are not

(deffunction find-leftmost-node-image-not-written (?card) 
  (bind ?*leftmost-coordinate* 10000)
  (bind ?*leftmost-node-image* FALSE)
  (bind ?node-image (get-first-card-node-image ?card))
  (while (neq ?node-image -1)
    (bind ?image-x (get-image-x ?card ?node-image))
    (if (and (or (< ?image-x ?*leftmost-coordinate*) 
  ;; If the image is to the left of the leftmost image found so far ...
                 (and (eq ?image-x ?*leftmost-coordinate*)
                      (bind ?image-y (get-image-y ?card ?node-image))
                      (< ?image-y (get-image-y ?card ?*leftmost-node-image*))
                 )
             )
  ;; ... or it is immediately above the leftmost image found so far ...
             (not (member ?node-image ?*node-images-written*))
  ;; ... and it hasn't already been written out ...
        )
     then (bind ?*leftmost-coordinate* ?image-x)
          (bind ?*leftmost-node-image* ?node-image)
    )
  ;; ... then instantiate it to be the leftmost so far.
    (bind ?node-image (get-next-card-node-image))
  )
  (return ?*leftmost-node-image*)          
)


(deffunction write-io (?card ?filename ?attribute ?list)
  (yield)
  (printout ?filename "<LI><B>" ?attribute ":</B> ")
  (bind ?counter 1)
  (bind ?length (length ?list))
  (while (<= ?counter ?length)  
    (bind ?arc (nth ?counter ?list))
    (bind ?name-attrib (get-first-object-attribute ?card ?arc))
    (if (eq ?name-attrib "type") 
     then (bind ?name-attrib (get-next-object-attribute))
    )    
    (bind ?arc-name (convert-to-html (get-object-string-attribute ?card ?arc ?name-attrib)))
    (if (> ?counter 1) then (printout ?filename ", "))
    (if (neq ?arc-name "") ;; i.e. if an attribute exists and is defined
     then (printout ?filename (convert-to-html ?arc-name))
     else (printout ?filename (get-object-string-attribute ?card ?arc "type"))
    )
    (bind ?counter (+ ?counter 1))
  )
  (printout ?filename crlf)
)

(deffunction write-arcs (?card ?filename ?list)
  (yield)
  (bind ?counter 1)
  (bind ?length (length ?list))
  (while (<= ?counter ?length)  
    (bind ?arc (nth ?counter ?list))
    (bind ?name-attrib (get-first-object-attribute ?card ?arc))
    (if (eq ?name-attrib "type") 
     then (bind ?name-attrib (get-next-object-attribute))
    )    
    (bind ?arc-name (convert-to-html (get-object-string-attribute ?card ?arc ?name-attrib)))
    (if (neq ?arc-name "") ;; i.e. if the "name" attribute exists and is defined
     then (printout ?filename "<LI><I>" (convert-to-html ?arc-name) "</I>" crlf)
     else (printout ?filename "<LI><I>" Arc "</I>" crlf)
    )
    (printout ?filename "<UL>" crlf)
    (bind ?attribute (get-first-object-attribute ?card ?arc))
    (while (neq ?attribute "")
     (if  (neq ?attribute ?name-attrib)
      then    (printout ?filename "<LI><B>" ?attribute ":</B> "
               (convert-to-html (get-object-string-attribute ?card ?arc ?attribute)) crlf)
     )
     (bind ?attribute (get-next-object-attribute))
    )
    (printout ?filename "</UL>" crlf crlf)
    (bind ?counter (+ ?counter 1))
  )
)
      
(deffunction write-input-output (?card ?filename ?node ?node-image)
  (bind ?*to-list* (mv-append))
  (bind ?*from-list* (mv-append))
  (bind ?arc-image (get-first-node-image-arc ?card ?node-image))
  (while (neq ?arc-image -1)
    (bind ?to (get-arc-image-to ?card ?arc-image))    
    (bind ?from (get-arc-image-from ?card ?arc-image)) 
    (if (eq ?node-image ?to)
     then (bind ?*to-list* (mv-append ?*to-list* (get-object-from-image ?card ?arc-image)))
     else (if (eq ?node-image ?from)
           then (bind ?*from-list* (mv-append ?*from-list*
(get-object-from-image ?card ?arc-image)))
           else (printout t "-- Error: Arc doesn't link to node" crlf)
          )
    )
    (bind ?arc-image (get-next-node-image-arc))
  )
  (write-io ?card ?filename "Inputs" ?*to-list*)
  (write-io ?card ?filename "Outputs" ?*from-list*)
)

;; This function makes sure expansion cards are written out in the same order
;; as the nodes in the diagram

(deffunction note-expansion-cards (?card ?node-image)  
  (bind ?expansion (get-first-image-expansion ?card ?node-image))
  (while (neq ?expansion -1)
    (bind ?*expansion-cards* (mv-append ?*expansion-cards* ?expansion))
    (bind ?expansion (get-next-image-expansion))
  )
)

(deffunction write-nodes-and-arcs-from-left (?card ?filename)
;  (set-status-message "Writing nodes and arcs...")
  (bind ?node-image (find-leftmost-node-image-not-written ?card))
  (while (neq ?node-image FALSE)
    (bind ?*node-images-written* (mv-append ?node-image ?*node-images-written*))
    (bind ?node (get-object-from-image ?card ?node-image))
    (note-expansion-cards ?card ?node-image)
    (bind ?node-name (convert-to-html (get-object-string-attribute ?card ?node "name")))
    (if (neq ?node-name "") ;; i.e. if the "name" attribute exists and is defined
     then (bind ?principal-name ?node-name)
     else (bind ?principal-name "Node")
    )
    (printout ?filename "<LI>")
    (trans-diagram-output-link ?filename ?card ?node-image TRUE) ; Possibly a start href
    (printout ?filename ?principal-name)
    (trans-diagram-output-link ?filename ?card ?node-image FALSE) ; Possibly an end href
    (printout ?filename crlf)

    (printout ?filename "<UL>" crlf)
    (bind ?attribute (get-first-object-attribute ?card ?node))
    (while (neq ?attribute "")
     (if  (neq ?attribute "name") 
      then    (printout ?filename "<LI><B>" (convert-to-html ?attribute) ":</B> "
               (convert-to-html (get-object-string-attribute ?card ?node ?attribute)) crlf)
     )
     (bind ?attribute (get-next-object-attribute))
    )
    (write-input-output ?card ?filename ?node ?node-image)
    (printout ?filename "</UL>" crlf crlf) 
    (write-arcs ?card ?filename ?*from-list*) 
    (bind ?node-image (find-leftmost-node-image-not-written ?card))
  )
  (bind ?*node-images-written* (mv-append))
)  

;;; Outputs am HTML link from this image, if any
(deffunction trans-diagram-output-link (?file ?card ?node-image ?start)
 (bind ?itemid (diagram-image-get-item ?card ?node-image))
 (if (> ?itemid -1) then
   (bind ?linkid (item-get-first-link ?card ?itemid))
   (if (> ?linkid -1) then
     (bind ?othercard (link-get-card-to ?linkid))
     (if (neq ?othercard ?card) then
      (trans-add-card ?othercard)
      (if ?start then
        (printout ?file
         (str-cat "<a href=\"" ?*trans-actual-path* "/" (trans-find-card-file ?othercard) "\">")
        )
        else
        (printout ?file "</a>")
      )
     )
    )
  )
)
