;------------------------------------------------------------------------------
; File:         expand.clp
; Author:	Jussi Stader
; Date:		7/12/94
; Connections:  clips, hardy, and windows utils
;
; Contents:     support for expansions of IDEF0 cards, 
;		IDEF0 numbering of card components
; Updates:
;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
; Instructions for providing these facilities in a HARDY application
;------------------------------------------------------------------------------
; make a loader file containing the following:
;
; (register-event-handler CustomMenu "IDEF0" idef0-event-handler)
; (load "/home/hardy/hardy/library/clputils/general/clips.clp")
; (load "/home/hardy/hardy/library/clputils/general/hardy.clp")
; (load "/home/hardy/hardy/library/clputils/general/windows.clp")
; (load "/home/hardy/hardy/library/clputils/specific/idef/idefutil.clp")
; (load "/home/hardy/hardy/library/clputils/specific/idef/expand.clp")
;
; make sure you have a similar function like this loaded too
; (load this last):
;
; (deffunction idef0-event-handler (?card-id ?option)
;   (if (eq ?option "Expand Selected Activity") then
;      (idef0-expand-task ?card-id) else
;    (if (eq ?option "Clear Selected Numbers") then
;       (idef0-clear-selected-numbers ?card-id) else
;     (if (eq ?option "Fill in Numbers") then
;        (idef0-fill-in-numbers ?card-id 0) else
;      (if (eq ?option "Set Numbers") then
;         (idef0-fill-in-numbers ?card-id 1) else
;    TRUE))))
; )
;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;       Notes and General Comments
;------------------------------------------------------------------------------
;
; this code heavily relies on the names of of IDEF0 nodes and arcs and their
; attributes.
;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------


;;;----------------------------------------------------------------------------
;;;			IDEF0 cards and expansions
;;;----------------------------------------------------------------------------

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;			expansion
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; (idef0-get-diagram-number ?card)
;;; Return: string, e.g. "A", "A34"
;; returns the IDEF0 number of a diagram (A1, A45). Returns "A" if no 
;; composable number is found (for A0, A-0, ...)  
(deffunction idef0-get-diagram-number (?card)
  (bind ?title (card-get-string-attribute ?card "title"))
  (bind ?length (str-index " " ?title))
  (if ?length then
    (bind ?number (sub-string 1 (- ?length 1) ?title))
   else (bind ?number "A"))
  (if (or (eq ?number "A0") (eq ?number "A00") (eq ?number "A-0")) then
     (return "A")
   else (return ?number)) )


;;; (idef0-make-node ?card ?type ?name ?number ?x ?y)
;;; Return: not to be used
;; make a node of the given type at the given position 
;; with the name and number in sensible attributes 
;; NOTE: expects ?number to be of type string 
(deffunction idef0-make-node (?card ?type ?name ?number ?x ?y)
  (if ?name then
    (bind ?image (node-image-create ?card ?type))
    (bind ?node (diagram-image-get-object ?card ?image))
    (bind ?att "name") ;; for Activity, Control, Mechanism
    (if (eq ?type "Input/Output") then (bind ?att "Data"))
    (diagram-object-set-string-attribute ?card ?node ?att ?name)
    (diagram-object-set-string-attribute ?card ?node "Number" ?number)
    (diagram-image-move ?card ?image ?x ?y)
    (diagram-object-format-text ?card ?node)
  else (format t "ERROR: no node data - not creating node of type %s%n" ?type))
)

;;; (idef0-expand-admin ?card ?new-card ?node-image)
;;; Return: not to be used
;; make an admin node for an IDEF0 diagram 
;; with inferred title and node attributes and 
;; asking for date and author if yet unknown (once per session) 
(deffunction idef0-expand-admin (?card ?new-card ?node-image)
  (bind ?number 
	(str-cat (idef0-get-diagram-number ?card)
	   (diagram-image-get-string-attribute ?card ?node-image "Number")))
  (bind ?title (get-card-string-attribute ?new-card "title"))
  (if (eq ?title "A diagram card") then (bind ?title ""))
  (card-set-string-attribute ?new-card "title"
		(str-cat ?number " :: " ?title))
  ;;; make admin node
  (bind ?image (node-image-create ?new-card "Admin"))
  (bind ?node (diagram-image-get-object ?new-card ?image))
  (diagram-object-set-string-attribute ?new-card ?node "title" ?title)
  (diagram-object-set-string-attribute ?new-card ?node "node" ?number)
  (diagram-object-set-string-attribute ?new-card ?node "date"
	(get-date ?new-card))
  (diagram-object-set-string-attribute ?new-card ?node "author"
	(get-author ?new-card))
  (diagram-image-move ?new-card ?image 70 30)
  (diagram-object-format-text ?new-card ?node)
)

;;; (idef0-expand-tasks ?new-card)
;;; Return: 2 field multivalue of numbers
;; make asked number of activity nodes on the new card and lay them out
(deffunction idef0-expand-tasks (?new-card)
  (bind ?string
       (get-choice "How many tasks/activities does the node expand into?"
		(create$ "2" "3" "4" "5" "6") 0 (card-get-frame ?new-card)))
  (if (neq ?string "") then
    (bind ?number (string-to-float ?string))
    (bind ?count 1)
    (bind ?x 80)
    (bind ?y 80)
    (while (<= ?count ?number) do
      (bind ?x (+ ?x 120))
      (bind ?y (+ ?y 80))
      (idef0-make-node ?new-card "Activity" "" (str-cat ?count) ?x ?y)
      (bind ?count (+ 1 ?count)))
   else (bind ?x 600)
  	(bind ?y 400))
  (return (create$ (+ ?x 100) (+ ?y 100)))
)

;;; (idef0-get-x ?node-type ?count)
;;; Return: number
;; get a good x-coordinate for the nth node of the given type 
(deffunction idef0-get-x (?node-type ?count)
 (if (eq ?node-type "Input/Output") then
   (return 50) else
   (return (+ 100 (* ?count 100))))
)
;;; (idef0-get-y ?node-type ?count ?max)
;;; Return: number
;; get a good y-coordinate for the nth node of the given type 
(deffunction idef0-get-y (?node-type ?count ?max)
 (if (eq ?node-type "Input/Output") then
   (return (+ 100 (* ?count 100))) else
  (if (eq ?node-type "Control") then (return 50) else
    (return ?max)))
)

;;; (idef0-get-arc-data ?card ?node-type ?arc-image ?from-to)
;;; Return: multivalue of 2 strings
;; get the data and number for a given arc-image. 
;; uses any data on datafow arcs (passing junctions) 
;; uses the from-node (skipping junctions) for the data for other arcs
(deffunction idef0-get-arc-data (?card ?node-type ?arc-image ?from-to)
  (bind ?arc (diagram-image-get-object ?card ?arc-image))
  (bind ?data "")
  (bind ?number (diagram-object-get-string-attribute ?card ?arc "Number"))
  (bind ?arc-type (diagram-object-get-string-attribute ?card ?arc "type"))
  (bind ?attribute (idef0-get-arc-type-data-attribute ?arc-type))
  (bind ?data (get-arc-image-attribute ?card ?arc-image ?from-to ?attribute))
  ;; no value on arc(s) so try node on other side
  (if (eq "" ?data) then
    (bind ?node (arc-image-get-node-object ?card ?arc-image ?from-to))
    (if (eq ?node FALSE) then (return (create$ FALSE FALSE)))
    (bind ?type (diagram-object-get-string-attribute ?card ?node "type"))
    (if (eq ?type "Input/Output") then
      (bind ?data (diagram-object-get-string-attribute ?card ?node "Data")) 
     else
      (bind ?data (diagram-object-get-string-attribute ?card ?node "name"))))
  (return (create$ ?data ?number))
)

;;; (idef0-type-expand-external ?card ?new-card ?node-type ?arcs ?max-y)
;;; Return: TRUE
;; for each given arc generate an external node of the right type 
;; (depending on arc-location)  at a good position
;; and fill it with a sensible name and number
(deffunction idef0-type-expand-external 
			(?card ?new-card ?node-type ?arcs ?max-y)
 (bind ?count 1)
 (bind ?data "")
 (while (<= ?count (length$ ?arcs)) do
   (bind ?node-spec 
	 (idef0-get-arc-data ?card ?node-type (nth$ ?count ?arcs) "from"))
   (idef0-make-node ?new-card ?node-type 
	(nth$ 1 ?node-spec) (nth$ 2 ?node-spec)
	(idef0-get-x ?node-type ?count) 
	(idef0-get-y ?node-type ?count ?max-y)) 
   (bind ?count (+ 1 ?count)))
  TRUE
)

;;; (idef0-expand-outdata ?card ?new-card ?arcs ?max-x)
;;; Return: not to be used
;; for each given output arc make node with right data at right location
(deffunction idef0-expand-outdata (?card ?new-card ?arcs ?max-x)
 (bind ?count 1)
 (bind ?data "")
 (while (<= ?count (length$ ?arcs)) do
   (bind ?node-spec 
	 (idef0-get-arc-data ?card "Input/Output" (nth$ ?count ?arcs) "to"))
   (idef0-make-node ?new-card "Input/Output" 
	(nth$ 1 ?node-spec) (nth$ 2 ?node-spec)
	?max-x (* ?count 100))
   (bind ?count (+ 1 ?count)))
)

;;; (idef0-expand-externals ?card ?new-card ?node-image ?max-pos)
;;; Return: not to be used
;; collect the node's arcs of different meaning (depending on node-side)
;; and make nodes of them on the new card.
(deffunction idef0-expand-externals (?card ?new-card ?node-image ?max-pos)
  (bind ?control   (create$))
  (bind ?indata    (create$))
  (bind ?outdata   (create$))
  (bind ?mechanism (create$))
  (bind ?arc-image (node-image-get-first-arc-image ?card ?node-image))
  (while (neq ?arc-image -1) do
    (bind ?other-node (arc-image-get-image-from ?card ?arc-image))
    (if (eq ?node-image ?other-node) then
       (bind ?outdata (append ?outdata ?arc-image))
     else (bind ?attach (arc-image-get-attachment-to ?card ?arc-image))
	  (if (eq ?attach 0) then
       	     (bind ?control (append ?control ?arc-image)) else
	   (if (eq ?attach 2) then
       	      (bind ?mechanism (append ?mechanism ?arc-image)) else
	    (if (eq ?attach 3) then
       	      (bind ?indata (append ?indata ?arc-image))))))
    (bind ?arc-image (node-image-get-next-arc-image)))
  (bind ?max-y (nth$ 2 ?max-pos))
  (idef0-type-expand-external ?card ?new-card "Input/Output" ?indata ?max-y)
  (idef0-expand-outdata  ?card ?new-card ?outdata (nth$ 1 ?max-pos))
  (idef0-type-expand-external ?card ?new-card "Control" ?control ?max-y)
  (idef0-type-expand-external ?card ?new-card "Mechanism" ?mechanism ?max-y)
)

;;; (idef0-expand-task ?card)
;;; Return: not to be used
;; for the selected node make an expansion card and fill it in with external
;; nodes and layed-out activities of a given number
(deffunction idef0-expand-task (?card)
  (bind ?node-image 
	(get-selected-type-image ?card "node" "Activity" "one"))
  (if (eq ?node-image (create$)) then
     (message-box "There is no activity node selected.
Plese select an activity and try again" OK 0 (card-get-frame ?card))

   else (bind ?new-card (diagram-card-create-expansion ?card ?node-image))
	(idef0-expand-admin ?card ?new-card ?node-image)
	(bind ?max-pos (idef0-expand-tasks ?new-card))
	(idef0-expand-externals ?card ?new-card ?node-image ?max-pos))
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;		numbering
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; (fill-activity-arc-numbers ?card ?arc-images ?side ?override)
;;; Return: not to be used
;; fill the number attribute according to position on node-side
;; NOTE: this uses x/y positions now, should be re-written
(deffunction fill-activity-arc-numbers (?card ?arc-images ?side ?override)
  ;;; for dataflow-type arcs this should check whether arc connects activities.
  ;;; If yes, use no-from and no-to for number
  (bind ?arc-images (sort-arc-images-by-position ?card ?arc-images ?side))
  (bind ?count 1)
  (while (<= ?count (length$ ?arc-images)) do
    (bind ?arc-image (nth$ ?count ?arc-images))
    (bind ?arc (diagram-image-get-object ?card ?arc-image))
    (if (and (eq ?override 0) 
	   (neq "" (diagram-object-get-string-attribute ?card ?arc "Number")))
     then T
     else (diagram-object-set-string-attribute 
		?card ?arc "Number" (str-cat ?count))
          (diagram-object-format-text ?card ?arc))
    (bind ?count (+ 1 ?count)))
)


;;; (fill-activity-numbers ?card ?node-images ?override)
;;; Return: not to be used
(deffunction fill-activity-numbers (?card ?node-images ?override)
  (bind ?node-images (sort-node-images-by-position ?card ?node-images))
  (bind ?count 1)
  (while (<= ?count (length$ ?node-images)) do
    (bind ?node-image (nth$ ?count ?node-images))
    (bind ?node (diagram-image-get-object ?card ?node-image))
    (if (and (eq ?override 0)
	   (neq "" (diagram-object-get-string-attribute ?card ?node "Number")))
     then T
     else (diagram-object-set-string-attribute 
			?card ?node "Number" (str-cat ?count))
	  (diagram-object-format-text ?card ?node))
    (bind ?control   (create$))
    (bind ?indata    (create$))
    (bind ?outdata   (create$))
    (bind ?mechanism (create$))
    (bind ?arc-image (node-image-get-first-arc-image ?card ?node-image))
    ;; collect arcs by input-side
    (while (neq ?arc-image -1) do
      (bind ?other-node (arc-image-get-image-from ?card ?arc-image))
      (if (eq ?node-image ?other-node) then
         (bind ?outdata (append ?outdata ?arc-image))
       else (bind ?attach (arc-image-get-attachment-to ?card ?arc-image))
	    (if (eq ?attach 0) then
               (bind ?control (append ?control ?arc-image)) else
	     (if (eq ?attach 2) then
       	       (bind ?mechanism (append ?mechanism ?arc-image)) else
	    (if (eq ?attach 3) then
     	       (bind ?indata (append ?indata ?arc-image))))))
      (bind ?arc-image (node-image-get-next-arc-image)))
    ;;; for each arc-type, number arcs according to position
    (fill-activity-arc-numbers ?card ?control 0 ?override)
    (fill-activity-arc-numbers ?card ?indata 3 ?override)
    (fill-activity-arc-numbers ?card ?outdata 1 ?override)
    (fill-activity-arc-numbers ?card ?mechanism 2 ?override)
    (bind ?count (+ 1 ?count)))
)

;;; (fill-control-numbers ?card ?control ?override)
;;; Return: 
(deffunction fill-control-numbers (?card ?control ?override)
)
;;; (fill-mechanism-numbers ?card ?mechanism ?override)
;;; Return: 
(deffunction fill-mechanism-numbers (?card ?mechanism ?override)
)
;;; (fill-inout-numbers ?card ?inout ?override)
;;; Return: 
(deffunction fill-inout-numbers (?card ?inout ?override)
)


;;; (idef0-fill-in-numbers ?card ?override)
;;; Return: TRUE
(deffunction idef0-fill-in-numbers (?card ?override)
  (bind ?control   (create$))
  (bind ?inout     (create$))
  (bind ?activity  (create$))
  (bind ?mechanism (create$))
  (bind ?node-image (diagram-card-get-first-node-image ?card))
   ;; go through node images to find first selected node of right type
  (while (neq ?node-image -1) do
;    (bind ?node (diagram-image-get-object ?card ?node-image))
    (bind ?type (diagram-image-get-string-attribute ?card ?node-image "type"))
    (if (eq ?type "Activity") then
       (bind ?activity (add-to-sorted-by-position ?card ?node-image ?activity 
	  (diagram-image-get-x ?card ?node-image) 
	  (diagram-image-get-y ?card ?node-image))) else
     (if (eq ?type "Input/Output") then
        (bind ?inout (append ?inout ?node-image)) else
      (if (eq ?type "Control") then
         (bind ?control (append ?control ?node-image)) else
       (if (eq ?type "Mechanism") then
          (bind ?mechanism (append ?mechanism ?node-image)) else
        (if (eq ?type "Admin") then
           (bind ?admin ?node-image))))))
    (bind ?node-image (diagram-card-get-next-node-image)))
  (fill-control-numbers   ?card ?control ?override)
  (fill-mechanism-numbers ?card ?mechanism ?override)
  (fill-inout-numbers     ?card ?inout ?override)
  (fill-activity-numbers  ?card ?activity ?override)
  TRUE
)



;;; (idef0-clear-selected-numbers ?card)
;;; Return: TRUE or OK
(deffunction idef0-clear-selected-numbers (?card)
 (bind ?selected FALSE)
  ;; go through node images to update selected nodes
 (bind ?image (diagram-card-get-first-node-image ?card))
 (while (neq ?image -1) do
   (if (eq 1 (diagram-image-selected ?card ?image))	;; selected
    then (bind ?selected T)
	 (bind ?object (diagram-image-get-object ?card ?image))
	 (diagram-object-set-string-attribute ?card ?object "Number" "")
         (diagram-object-format-text ?card ?object))
   (bind ?image (diagram-card-get-next-node-image)))

  ;; go through arc images to update selected arcs
 (bind ?image (diagram-card-get-first-arc-image ?card))
 (while (neq ?image -1) do
   (if (eq 1 (diagram-image-selected ?card ?image))	;; selected
    then (bind ?selected T)
	 (bind ?object (diagram-image-get-object ?card ?image))
	 (diagram-object-set-string-attribute ?card ?object "Number" "")
         (diagram-object-format-text ?card ?object))
   (bind ?image (diagram-card-get-next-arc-image)))

 (if (eq ?selected FALSE) then
        (no-selection-message ?card)
  else TRUE)
)


