
;;; CONTENTS:

;;; Miscellaneous functions 
;;; Attribute related functions
;;; FIND functions
;;; Card functions 
;;; String & list functions
;;; Diagram-related functions
;;; Arcs and linked nodes functions
;;; Object-related functions


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Miscellaneous functions 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



(deffunction rand (?n)
  (+ 1 (mod (random) ?n))
)


(deffunction get-name (?card-id ?object)
  (get-object-string-attribute ?card-id ?object "Id")
)

(deffunction get-object-type (?card-id ?object)
  (get-object-string-attribute ?card-id ?object "type"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Attribute related functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Add to a (multi-valued) string attribute, by concatenating to the string
(deffunction append-object-string-attribute (?card-id ?object ?att ?string)
  (set-object-string-attribute ?card-id ?object ?att
     (str-cat  (get-object-string-attribute ?card-id ?object ?att) " \""
?string "\"")
  )
)

;;; Delete a value from a multi-valued string attribute
(deffunction delete-object-mv-string-attribute (?card-id ?object ?att ?value)
  (bind ?old-list (str-explode (get-object-string-attribute ?card-id ?object
?att)))
  (bind ?index (member ?value ?old-list))
  (if (integerp ?index)
   then
  (set-object-string-attribute ?card-id ?object ?att
     (str-implode (mv-delete ?index ?old-list))
  )
   else (get-text-from-user (str-cat "Error: " ?value " is not in " ?att " of
" (get-object-string-attribute ?card-id ?object "Id")) "")
  )
)
              
;;; Get all values of a multi-valued string attribute as a multi-valued list  
(deffunction get-object-mv-string-attribute (?card-id ?object ?att)
  (if (and (neq ?object -1) (neq ?att ""))
   then (str-explode (get-object-string-attribute ?card-id ?object ?att))
   else (mv-append)
  )
)


(deffunction select-from-mv-attribute (?n ?list)
   (bind ?val (nth ?n ?list))
   (bind ?exploded-val (nth 1 (str-explode ?val))) ;; same code as STR-TO_INTEGER
   (if (numberp ?exploded-val)
    then (return ?exploded-val)
    else (return ?val)
   )
)

(deffunction attribute-exists (?card-id ?node ?attribute)
  (bind ?att (get-first-object-attribute ?card-id ?node))
    (while (neq ?att "") do
      (if (eq ?att ?attribute) 
       then (return TRUE)
       else (bind ?att (get-next-object-attribute))
      )
    )
    (return FALSE)
)


;;; This function takes a multi-valued string attribute where the values are
;;; separated by new lines and converts it into a list of values

(deffunction mv-string-att-to-list (?card ?node ?att)
  (bind ?values-string (get-object-string-attribute ?card ?node ?att))
  (bind ?values (create$))
  (bind ?index (str-index "
" ?values-string))
  (while (neq ?index FALSE) do
    (bind ?len (length ?values-string))
    (bind ?first-val (sub-string 1 (- ?index 1) ?values-string))
    (bind ?values (create$ ?first-val ?values))
    (bind ?values-string (sub-string (+ ?index 1) ?len ?values-string))
    (bind ?index (str-index "
" ?values-string))
  )
  (bind ?values (create$ ?values ?values-string )) ;; add the last one
  (return ?values)
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; FIND functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;; Given the value of an object's "Id" attribute, find the object.
;;; Requires the object to have "Id" as an attribute
;; WARNING: This uses GET-FIRST-CARD-NODE and so may not be called
;;; within any other call to GET-FIRST-CARD-NODE ... GET-NEXT-CARD-NODE
(deffunction find-object-from-id (?card-id ?id)
  (bind ?node (get-first-card-node ?card-id))
  (while (neq ?node -1)
         (if (eq ?id (get-object-string-attribute ?card-id ?node "Id"))
          then (return ?node)
          else  (bind ?node (get-next-card-node))
         )
  )
  (return -1)
)


;;; Same as above except it uses GET-FIRST-CARD-NODE-IMAGE
(deffunction find-object-from-id-on-exp-card (?card-id ?name)
  (bind ?node-image (get-first-card-node-image ?card-id))
  (while (neq ?node-image -1)
         (bind ?node (get-object-from-image ?card-id ?node-image))
         (if (eq ?name (get-object-string-attribute ?card-id ?node "Id"))
          then (return ?node)
          else  (bind ?node-image (get-next-card-node-image))
         )
  )
  (return -1)
)


;; Uses GET-FIRST-CARD-NODE *and* GET-FIRST-CARD-NODE-IMAGE
(deffunction find-object-from-id-on-card (?card-id ?name)
  (bind ?node (get-first-card-node ?card-id))
  (while (neq ?node -1)
         (if (eq ?name (get-object-string-attribute ?card-id ?node "Id"))
          then (if (neq (get-first-card-node-image ?card-id ?node) -1)
                then (return ?node)
                else (bind ?node (get-next-card-node))
               )
          else  (bind ?node (get-next-card-node))
         )
  )
  (return -1)
)

;; Uses GET-FIRST-CARD-NODE-IMAGE
(deffunction find-image-from-name-on-card (?card-id ?name)
  (bind ?node-image (get-first-card-node-image ?card-id))
  (while (neq ?node-image -1)
         (if (eq ?name 
               (get-object-string-attribute ?card-id 
                  (get-object-from-image ?card-id ?node-image) 
                  "Id"))
          then (return ?node-image)
          else (bind ?node-image (get-next-card-node-image))
         )
  )
  (return -1)
)


;;; Given an attribute, find an object with that attribute
;; WARNING: This uses GET-FIRST-CARD-NODE and so may not be called within any
;; other call to GET-FIRST-CARD-NODE ... GET-NEXT-CARD-NODE
(deffunction find-object-from-attribute (?card-id ?attribute)
  (bind ?node (get-first-card-node ?card-id))
  (while (neq ?node -1)
         (bind ?att (get-first-object-attribute ?card-id ?node))
         (while (neq ?att "")
          (if (eq ?att ?attribute)
           then (return ?node)
           else  (bind ?att (get-next-object-attribute))
          )
         )
    else  (bind ?node (get-next-card-node))
   )
)

;; Find an object with a given attribute & value
;; WARNING: This uses GET-FIRST-CARD-NODE and so may not be called within any
;; other call to GET-FIRST-CARD-NODE ... GET-NEXT-CARD-NODE
(deffunction find-object-from-attribute-value (?card-id ?attribute ?value)
  (bind ?node (get-first-card-node ?card-id))
  (while (neq ?node -1)
         (if (eq ?value (get-object-string-attribute ?card-id ?node
?attribute))             
           then (return ?node)
          )
           else  (bind ?node (get-next-card-node))
   )
)


;;; Find an attribute in an object which contains a given value
(deffunction find-attribute-from-value (?card-id ?node ?value)
  (bind ?att (get-first-object-attribute ?card-id ?node))
  (while (neq ?att "")
          (if (or (eq (get-object-string-attribute ?card-id ?node ?att) ?value)
                  (member ?value (get-object-mv-string-attribute ?card-id ?node ?att))
              )
           then (return ?att)
                (bind ?att "")
           else  (bind ?att (get-next-object-attribute))
          )
  )
  (return "") ;; failure
)





;; This could be made more efficient. Not currently recommended for use.
(deffunction find-typed-object-from-attribute-value (?card-id ?type ?attribute ?value)
  (bind ?node (get-first-card-node ?card-id))
  (while (neq ?node -1)
         (if (eq ?type (get-object-type ?card-id ?node))
          then
           (bind ?att (get-first-object-attribute ?card-id ?node))
           (while (neq ?att "")
            (if (and (eq ?att ?attribute)
                     (eq ?value (get-object-string-attribute ?card-id ?node
?attribute)))
                   then (return ?node)
             else  (bind ?att (get-next-object-attribute))
            )
          )
    else  (bind ?node (get-next-card-node))
         )
  )
  (return -1)
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Card functions 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



(deffunction get-parent (?card-id)
  (bind ?item (get-card-special-item ?card-id))
  (bind ?link (get-first-item-link ?card-id ?item))
  (while (neq ?link -1) do
   (bind ?from (get-link-card-from ?link))
   (if (neq ?from ?card-id)
    then (return ?from)
    else
   (bind ?link (get-next-item-link))
   )
  )
;;  (return -1)
)

(deffunction get-first-child-card (?card-id)
  (bind ?item (get-card-special-item ?card-id))
  (bind ?link (get-first-item-link ?card-id ?item))
  (while (neq ?link -1) do
   (bind ?to (get-link-card-to ?link))
   (if (neq ?to ?card-id)
    then (return ?to)
    else
   (bind ?link (get-next-item-link))
   )
  )
)

(deffunction get-all-child-cards (?card-id)
  (bind ?cards (mv-append))
  (bind ?item (get-card-special-item ?card-id))
  (bind ?link (get-first-item-link ?card-id ?item))
  (while (neq ?link -1) do
   (bind ?to (get-link-card-to ?link))
   (if (neq ?to ?card-id)
    then (bind ?cards (mv-append ?cards ?to))
   )
   (bind ?link (get-next-item-link))
  )
  (bind ?node-image (get-first-card-node-image ?card-id))
  (while (neq ?node-image -1) do
   (bind ?item (get-image-item ?card-id ?node-image))
   (bind ?link (get-first-item-link ?card-id ?item))
   (while (neq ?link -1) do
     (bind ?to (get-link-card-to ?link))
     (if (neq ?to ?card-id)
      then (bind ?cards (mv-append ?cards ?to))
     )
      (bind ?link (get-next-item-link))
   )
   (bind ?node-image (get-next-card-node-image))
  )
  (return ?cards)
)

(deffunction get-title (?card-id)
  (get-card-string-attribute ?card-id "title")
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; String & list functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



;; Given a string of an integer e.g. "25", return the integer 
(deffunction str-to-integer (?string)
  (return (nth 1 (str-explode ?string)))
)



;; Uses get-node-image rather than get-node so that it differentiates between
;; nodes on expansion cards

(deffunction build-node-list-on-card (?card-id ?node-image)
  (if (neq ?node-image -1)
   then (bind ?node (get-object-from-image ?card-id ?node-image))
        (bind ?new-node-image (get-next-card-node-image))
        (mv-append (get-object-string-attribute ?card-id ?node "Id")
                   (build-node-list-on-card ?card-id ?new-node-image))
   else (return (mv-append FALSE))  ;; error prevention
  )
)

(deffunction make-node-list-on-card (?card-id)
  (bind ?node-image (get-first-card-node-image ?card-id))
  (if (neq ?node-image -1)
   then (bind ?raw-list (build-node-list-on-card ?card-id ?node-image))
   (mv-delete (member FALSE ?raw-list) ?raw-list)
   else (mv-append)
  )
)



(deffunction build-node-list (?card-id ?node)
  (if (neq ?node -1)
   then (bind ?new-node (get-next-card-node))
        (mv-append (get-object-string-attribute ?card-id ?node "Id")
                   (build-node-list ?card-id ?new-node))
   else (return (mv-append FALSE))  ;; error prevention
  )
)


(deffunction make-node-list (?card-id)
  (bind ?node (get-first-card-node ?card-id))
  (if (neq ?node -1)
   then (bind ?raw-list (build-node-list ?card-id ?node))
   (mv-delete (member FALSE ?raw-list) ?raw-list)
   else (mv-append)
  )
)

;; This function includes an allowance for subcategories

(deffunction build-typed-node-list-on-card (?card-id ?node-image ?type)
  (if (neq ?node-image -1)
   then (bind ?node (get-object-from-image ?card-id ?node-image))
        (bind ?new-node-image (get-next-card-node-image))
        (bind ?node-type (get-object-type ?card-id ?node))
        (if (eq ?node-type ?type)
         then
        (mv-append (get-object-string-attribute ?card-id ?node "Id")
                   (build-typed-node-list-on-card ?card-id ?new-node-image ?type))
         else (mv-append (build-typed-node-list-on-card ?card-id ?new-node-image ?type))
        )
   else (return (mv-append FALSE))  ;; error prevention
  )
)


(deffunction make-typed-node-list-on-card (?card-id ?type)
  (bind ?node-image (get-first-card-node-image ?card-id))
  (if (neq ?node-image -1)
   then (bind ?raw-list (build-typed-node-list-on-card ?card-id ?node-image ?type))
   (mv-delete (member FALSE ?raw-list) ?raw-list)
   else (mv-append)
  )
)

(deffunction build-typed-node-list (?card-id ?node ?type)
  (if (neq ?node -1)
   then (bind ?new-node (get-next-card-node))
        (bind ?node-type (get-object-type ?card-id ?node))
        (bind ?id (get-object-string-attribute ?card-id ?node "Id"))
        (if (and (eq ?node-type ?type) (neq ?id ""))
         then (create$ ?id (build-typed-node-list ?card-id ?new-node ?type))
         else (create$ (build-typed-node-list ?card-id ?new-node ?type))
        )
   else (return (create$ FALSE))  ;; error prevention
  )
)


(deffunction make-typed-node-list (?card-id ?type)
  (bind ?node (get-first-card-node ?card-id))
  (if (neq ?node -1)
   then (bind ?raw-list (build-typed-node-list ?card-id ?node ?type))
   (mv-delete (member FALSE ?raw-list) ?raw-list)
   else (mv-append)
  )
)


;; used by MOVE-ITEM-TO-SUBCATEGORY to exclude subcategories themselves

(deffunction build-node-list-excluding-type (?card-id ?node-image ?type)
  (if (neq ?node-image -1)
   then (bind ?node (get-object-from-image ?card-id ?node-image))
        (bind ?new-node-image (get-next-card-node-image))
        (bind ?node-type (get-object-type ?card-id ?node))
        (if (neq ?node-type ?type)
         then
        (mv-append (get-object-string-attribute ?card-id ?node "Id")
                   (build-node-list-excluding-type ?card-id ?new-node-image ?type))
         else (mv-append (build-node-list-excluding-type ?card-id ?new-node-image ?type))
        )
   else (return (mv-append FALSE))  ;; error prevention
  )
)


(deffunction make-node-list-excluding-type (?card-id ?type)
  (bind ?node-image (get-first-card-node-image ?card-id))
  (if (neq ?node-image -1)
   then (bind ?raw-list (build-node-list-excluding-type ?card-id ?node-image ?type))
   (mv-delete (member FALSE ?raw-list) ?raw-list)
   else (mv-append)
  )
)





;;; This utility operates GET-CHOICE on a list of nodes on a card. It has 2 
;;; special functions:
;;;   1/ All expansion cards will appear as walking menus
;;;   2/ A list of extra options (created using MV-APPEND) will be attached to
;;;      the top of the list

(deffunction choose-from-node-list-plus-card (?card-id ?prompt ?options) 
 (bind ?node-list (mv-append ?options (string-sort (make-node-list ?card-id))))
  (bind ?exp-card (get-first-expansion-descendant ?card-id))
  (while (neq ?exp-card -1) do
   (if (neq ?exp-card ?card-id) then ;; BUG: get-first-expansion-descendant on
                                     ;; a leaf card returns itself!
    (bind ?title (get-title ?exp-card))
    (if (integerp (member ?title ?node-list))
     then (bind ?node-list (mv-delete (member ?title ?node-list) ?node-list))
    )
    (bind ?node-list (mv-append (str-cat ?title " =>") ?node-list))
  )
    (bind ?exp-card (get-next-expansion-descendant))
  )
  (bind ?choice (get-choice ?prompt ?node-list))
  (if (not (str-index " =>" ?choice)) ;; i.e. an expansion card is not chosen
   then (return (mv-append ?card-id ?choice))
   else (bind ?exp-card (find-card-by-title 
                           (sub-string 1 
                                       (- (str-length ?choice) 3)
                                       ?choice)))
        (choose-from-node-list-plus-card ?exp-card ?prompt ?options)
  )
)


(deffunction choose-from-node-list-no-expansion (?card-id ?prompt ?options) 
 (bind ?node-list (mv-append ?options (string-sort (make-node-list ?card-id))))
  (bind ?choice (get-choice ?prompt ?node-list))
  (return ?choice)
)

(deffunction choose-from-typed-node-list (?card-id ?prompt ?type ?options) 
 (bind ?node-list (mv-append ?options (string-sort (make-typed-node-list
?card-id ?type))))
  (bind ?choice (get-choice ?prompt ?node-list))
  (return ?choice)
)

;; returns the name of a node on the card
(deffunction choose-from-node-list (?card-id ?prompt ?options) 
  (return (nth 2 (choose-from-node-list-plus-card ?card-id ?prompt ?options)))
)

(deffunction mv-list-to-name-string (?card-id ?objects)
  (bind ?counter 1)
  (bind ?len (length ?objects))
  (bind ?string-list "")
  (while (< ?counter ?len) do ;; < not <= - see end of function
     (if (eq ?counter 1) 
      then
     (bind ?string-list (get-name ?card-id (nth ?counter ?objects)))
      else
     (bind ?string-list (str-cat ?string-list ", " 
                                  (get-name ?card-id (nth ?counter ?objects))))
     )
     (bind ?counter (+ ?counter 1))
  )
  (bind ?string-list (str-cat ?string-list " and " 
                                  (get-name ?card-id (nth ?len ?objects))))
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Diagram-related functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;; The programmatic equivalent of Refresh Display
(deffunction refresh-card (?card)
  (clear-card-canvas ?card)
  (redraw-diagram ?card)
)



(deffunction my-get-object-from-image (?card ?node ?root)
  (if (eq (get-object-from-image ?card ?node) -1)
   then (get-object-from-image ?root ?node)
  )
)



(deffunction my-create-node-image (?card-id ?node)
  (if (eq (is-card-shown ?card-id) 0)
   then (show-card ?card-id TRUE)
  )
  (create-node-image ?card-id ?node)
)

;; assumes name not used before

(deffunction do-create-new-node-image (?card-id ?type ?name ?x ?y)
   (bind ?root-card (find-diagram-root ?card-id))
   (bind ?new-image (my-create-node-image ?card-id ?type))
   (bind ?new-node (get-object-from-image ?card-id ?new-image))
   (set-object-string-attribute ?card-id ?new-node "Id" ?name)
   (move-image ?card-id ?new-image (+ 100 (rand 10)) (+ 100 (rand 10)))
   (format-object-text ?card-id ?new-node) 
;;   (refresh-card ?card-id)
   (return ?new-image)
)

(deffunction create-new-node-image (?card-id ?type ?name)
  (do-create-new-node-image ?card-id ?type ?name (+ 100 (rand 10)) 
                                               (+ 100 (rand 10)))
)

(deffunction create-new-node-image-with-coords (?card-id ?type ?name ?x ?y)
  (do-create-new-node-image ?card-id ?type ?name ?x ?y)
)



(deffunction delete-all-typed-images (?card-id ?type)
  (bind ?image (get-first-card-node-image ?card-id))
  (while (neq ?image -1) do
    (bind ?image-type (get-object-type ?card-id (get-object-from-image
?card-id ?image)))
    (if (eq ?image-type ?type)
     then (delete-image ?card-id ?image)
    )
    (bind ?image (get-next-card-node-image))
  )
)

;; WARNING: this calls GET-FIRST-CARD-NODE
;; Will find the first selected node

(deffunction get-selected-node (?card-id)
  (bind ?node (get-first-card-node ?card-id))
   (while (neq ?node -1) do
      (bind ?image (get-first-object-image ?card-id ?node))
      (if (eq 1 (image-selected ?card-id ?image))
             then (return ?node)
       else
      (bind ?node (get-next-card-node))
      )
   )
  (return -1)
)



(deffunction duplicate-hyperlinks (?old-card ?old-image ?new-card ?new-image)
  (bind ?old-item (get-image-item ?old-card ?old-image))
  (bind ?new-item (get-image-item ?new-card ?new-image))
  (bind ?link (get-first-item-link ?old-card ?old-item))
  (while (neq ?link -1) do
    (bind ?from (get-link-from ?link))
    (bind ?to (get-link-to ?link))
    (if (eq ?from ?old-item)
     then (link-items ?new-card ?new-item (get-link-card-to ?link) ?to)
     else (link-items (get-link-card-from ?link) ?from ?new-card ?new-item)
    )
    (bind ?link (get-next-item-link))
  )
)




;; Only duplicates nodes
;; WARNING: Uses GET-FIRST-CARD-NODE-IMAGE
(deffunction duplicate-all-nodes (?old-card ?new-card)
  (bind ?node-image (get-first-card-node-image ?old-card))
  (while (neq ?node-image -1) do
   (bind ?node (get-object-from-image ?old-card ?node-image))
   (bind ?new-image (duplicate-node-image ?new-card ?node))
   (duplicate-hyperlinks ?old-card ?node-image ?new-card ?new-image)
   (move-image ?new-card ?new-image (get-image-x ?old-card ?node-image)
                                    (get-image-y ?old-card ?node-image))
   (format-object-text ?old-card ?node) 
   (bind ?node-image (get-next-card-node-image))
  )
;;  (refresh-card ?new-card)
)



;; Only duplicates nodes
;; WARNING: Uses GET-FIRST-CARD-NODE-IMAGE
(deffunction duplicate-all-typed-nodes (?old-card ?type ?new-card)
  (bind ?node-image (get-first-card-node-image ?old-card))
  (while (neq ?node-image -1) do
   (bind ?node (get-object-from-image ?old-card ?node-image))
   (if (eq ?type (get-object-string-attribute ?old-card ?node "type"))
    then
    (bind ?new-image (duplicate-node-image ?new-card ?node))
    (duplicate-hyperlinks ?old-card ?node-image ?new-card ?new-image)
    (move-image ?new-card ?new-image (get-image-x ?old-card ?node-image)
                                    (get-image-y ?old-card ?node-image))
   )
    (format-object-text ?old-card ?node) 
  (bind ?node-image (get-next-card-node-image))
  )
;;  (refresh-card ?new-card)
)


;; Move a node to another card in the same expansion hierarchy
(deffunction move-node-to-card (?old-card ?node-name ?new-card)
  (bind ?node (find-object-from-id ?old-card ?node-name)) 
  (bind ?new-image (duplicate-node-image ?new-card ?node))
  (bind ?old-image (get-first-object-image ?old-card ?node))
  (duplicate-hyperlinks ?old-card ?old-image ?new-card ?new-image)
  (move-image ?new-card ?new-image (get-image-x ?old-card ?old-image)
                                   (get-image-y ?old-card ?old-image))
  (format-object-text ?old-card ?node) ; bug/feature, must use root card
  (delete-image ?old-card (get-first-object-image ?old-card ?node))
  (refresh-card ?old-card)
;;  (refresh-card ?new-card)
)

;; used in LADDER.CLP & DECTREE.CLP
(deffunction create-unique-new-image (?card-id ?new-name ?type)
  (if (eq (find-object-from-id ?card-id ?new-name) -1)
   then
   (bind ?new-image (my-create-node-image ?card-id ?type))
   (bind ?new-node (get-object-from-image ?card-id ?new-image))
   (set-object-string-attribute ?card-id ?new-node "Id" ?new-name)
   (move-image ?card-id ?new-image 40 20)
   (format-object-text ?card-id ?new-node)
   (return ?new-image)
   else 
   (get-text-from-user (str-cat "Error: There is already a " ?type " called "
                                    ?new-name) )
   (return "")
   )
)




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Arcs and linked nodes functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Find the first arc IMAGE going out of a node image
;; WARNING: This uses GET-FIRST-CARD-ARC_IMAGE and so may not be called within 
;; any other call to GET-FIRST-CARD-ARC-IMAGE ... GET-NEXT-CARD-ARC-IMAGE
(deffunction find-arc-from-image (?card ?image)
  (bind ?arc-image (get-first-card-arc-image ?card))
  (while (neq ?arc-image -1) do
    (bind ?arc-from (get-arc-image-from ?card ?arc-image))
    (if (eq ?arc-from ?image)
     then (return ?arc-image)
     else (bind ?arc-image (get-next-card-arc-image))
    )
  )
  (return "")
)


;; Find the first arc IMAGE going into a node image
;; WARNING: This uses GET-FIRST-CARD-ARC-IMAGE and so may not be called within 
;; any other call to GET-FIRST-CARD-ARC-IMAGE ... GET-NEXT-CARD-ARC-IMAGE
(deffunction find-arc-to-image (?card ?image)
  (bind ?arc-image (get-first-card-arc-image ?card))
  (while (neq ?arc-image -1) do
    (bind ?arc-to (get-arc-image-to ?card ?arc-image))
    (if (eq ?arc-to ?image)
     then (return ?arc-image)
     else (bind ?arc-image (get-next-card-arc-image))
    )
  )
)




;; Finds the node at the end of a specified TO link. 
(deffunction find-linked-node-to (?card-id ?arc-image)
  (bind ?node-image-to (get-arc-image-to ?card-id ?arc-image))
  (bind ?linked-node (get-object-from-image ?card-id ?node-image-to))
  (return ?linked-node)
)


;; Finds the node at the end of the first FROM link. Uses FIND-ARC-FROM-IMAGE
(deffunction find-first-linked-node-to (?card-id ?node)
  (bind ?arc-image (find-arc-from-image ?card-id (get-first-object-image
?card-id ?node)))
  (find-linked-node-to ?card-id ?arc-image)
)


;; Finds the node at the end of a specified FROM link. 
(deffunction find-linked-node-from (?card-id ?arc-image)
  (bind ?node-image-from (get-arc-image-from ?card-id ?arc-image))
  (bind ?linked-node (get-object-from-image ?card-id ?node-image-from))
  (return ?linked-node)
)

;; Finds the node at the end of the first TO link. Uses FIND-ARC-TO-IMAGE
(deffunction find-first-linked-node-from (?card-id ?node)
  (bind ?arc-image (find-arc-to-image ?card-id (get-first-object-image
?card-id ?node)))
  (find-linked-node-from ?card-id ?arc-image)
)


;; This could be made more efficient. Not currently recommended for use.
(deffunction find-typed-object-from-attribute-value (?card-id ?type ?attribute ?value)
  (bind ?node (get-first-card-node ?card-id))
  (while (neq ?node -1)
         (if (eq ?type (get-object-type ?card-id ?node))
          then
           (bind ?att (get-first-object-attribute ?card-id ?node))
           (while (neq ?att "")
            (if (and (eq ?att ?attribute)
                     (eq ?value (get-object-string-attribute ?card-id ?node
?attribute)))
                   then (return ?node)
             else  (bind ?att (get-next-object-attribute))
            )
          )
    else  (bind ?node (get-next-card-node))
         )
  )
  (return -1)
)





;; Find all arc IMAGEs going into a node image
;; WARNING: This uses GET-FIRST-CARD-ARC and so may not be called within any
;; other call to GET-FIRST-CARD-ARC ... GET-NEXT-CARD-ARC
(deffunction find-all-arcs-to-image (?card ?image)
  (bind ?arc-images (mv-append))
  (bind ?arc (get-first-card-arc ?card))
  (while (neq ?arc -1) do
    (bind ?arc-image (get-first-object-image ?card ?arc))
    (bind ?arc-to (get-arc-image-to ?card ?arc-image))
    (if (eq ?arc-to ?image)
     then (bind ?arc-images (mv-append ?arc-images ?arc-image))
    )
  (bind ?arc (get-next-card-arc))
  )
  (return ?arc-images)
)


;; WARNING: This uses GET-FIRST-CARD-ARC and so may not be called within any
;; other call to GET-FIRST-CARD-ARC ... GET-NEXT-CARD-ARC
(deffunction find-all-linked-nodes-to-image (?card ?image)
  (bind ?linked-nodes (mv-append))
  (bind ?arc (get-first-card-arc ?card))
  (while (neq ?arc -1) do
    (bind ?arc-image (get-first-object-image ?card ?arc))
    (bind ?arc-to (get-arc-image-to ?card ?arc-image))
    (if (eq ?arc-to ?image)
     then (bind ?linked-nodes 
           (mv-append ?linked-nodes (find-linked-node-from ?card ?arc-image)))
    )
  (bind ?arc (get-next-card-arc))
  )
  (return ?linked-nodes)
)

;; WARNING: This uses GET-FIRST-CARD-ARC and so may not be called within any
;; other call to GET-FIRST-CARD-ARC ... GET-NEXT-CARD-ARC
(deffunction find-all-linked-nodes-from-image (?card ?image)
  (bind ?linked-nodes (mv-append))
  (bind ?arc (get-first-card-arc ?card))
  (while (neq ?arc -1) do
    (bind ?arc-image (get-first-object-image ?card ?arc))
    (bind ?arc-from (get-arc-image-from ?card ?arc-image))
    (if (eq ?arc-from ?image)
     then (bind ?linked-nodes 
           (mv-append ?linked-nodes (find-linked-node-to ?card ?arc-image)))
    )
  (bind ?arc (get-next-card-arc))
  )
  (return ?linked-nodes)
)

(deffunction link-nodes (?card1 ?node1 ?card2 ?node2)
  (bind ?image1 (get-first-object-image ?card1 ?node1)) 
  (bind ?item1 (get-image-item ?card1 ?image1))
  (bind ?image2 (get-first-object-image ?card2 ?node2)) 
  (bind ?item2 (get-image-item ?card2 ?image2))
  (link-items ?card1 ?item1 ?card2 ?item2)
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Object-related functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defmessage-handler USER print-instance-msg ()
  (ppinstance)
)


(deffunction ppinst (?instance)
  (send ?instance print-instance-msg)
)

(deffunction get-object-id-list (?class)
  (bind ?instances (find-all-instances ((?i ?class)) TRUE))
  (bind ?id-list (create$))
  (bind ?len (length ?instances))
  (bind ?counter 1)
  (while (<= ?counter ?len) do
    (bind ?inst (nth ?counter ?instances))
    (bind ?id-list (create$ ?id-list (send ?inst get-id)))
    (bind ?counter (+ ?counter 1))
  )
  (return (string-sort ?id-list))
)  
