;;; ===========================================================================
;;; File:	clips.clp
;;; Author:	Jussi Stader
;;; Content:	generally useful CLIPS functions 
;;; Date:	06/12/94
;;;
;;; Update: 
;;; 		
;;; Comments: 
;;;  
;;;
;;; ===========================================================================

;;;----------------------------------------------------------------------------
;;; Things in this file:
;;;----------------------------------------------------------------------------
;;; (get-file)
;;; (get-file-load)
;;; (get-file-read)
;;; (newsym ?string)
;;; (append ?mv1 ?mv2)
;;; (add-if-not-member ?new ?set)
;;; (any-common-members ?mv1 ?mv2)
;;; (remove-all ?item ?set)
;;; (is-class-thing ?object ?class)
;;; (inst-name ?value)
;;; (value-format ?style ?file ?val)
;;; (write-values ?style ?ins ?in-length ?file)
;;; (write-list ?style ?ins ?in-length ?file)
;;; (make-string-value $?val)
;;; (last-char ?string)
;;; (strip-lead-whites ?string)
;;; (strip-trail-whites ?string)
;;; (strip-whites ?string)
;;; (capitalise-first ?string)
;;;----------------------------------------------------------------------------




;;;----------------------------------------------------------------------------
;;;			general
;;;----------------------------------------------------------------------------
; get-file, get-file-load, get-file-read, newsym(?string),

;;; (get-file)
;;; Return: TRUE or FALSE
;;; get a file name and open it for writing
(deffunction get-file ()
  (bind ?filename (file-selector "Enter a filename for writing"))
  (if (neq ?filename "") then
    (open ?filename file "w"))
)
;;; example: (if (get-file) then <do the stuff> else <no file given>

;;; (get-file-load)
;;; Return: TRUE or FALSE
;;; get a file name and load it
(deffunction get-file-load ()
  (bind ?filename (file-selector "Enter a filename for loading"))
  (if (neq ?filename "") then
    (load ?filename)
    (reset))			;;; *** this gets definstances to work
)
;;; example: (if (get-file) then <do the stuff> else <no file given>

;;; (get-file-read)
;;; Return: TRUE or FALSE
;;; get a file name and open it for reading
(deffunction get-file-read ()
  (bind ?filename (file-selector "Enter a filename for reading"))
  (if (neq ?filename "") then
    (open ?filename file "r"))
)
;;; example: (if (get-file) then <do the stuff> else <no file given>

;;; (newsym ?string)
;;; Return: symbol
;;; generates a new symbol that starts with the given string
(deffunction newsym (?string)
  (sym-cat ?string "-" (gensym))
)
;;; example: (make-instance (newsym "node") of NODE) ;  --> node-gen47


;;;----------------------------------------------------------------------------
;;;			multifield 
;;;----------------------------------------------------------------------------

;;; (append ?mv1 ?mv2)
;;; Return: multi-value
(deffunction append (?mv1 ?mv2)
  (create$ ?mv1 ?mv2)
)
;;; example: (append (create$ 1 2 3) 4) 		--> (1 2 3 4)
;;; 	     (append (create$ 1 2 3) (create$ 4 5)) 	--> (1 2 3 4 5)

;;; (add-if-not-member ?new ?set)
;;; Return: multi-value
;;; adds the first argument to the multifield variable, unless it is already in
(deffunction add-if-not-member (?new ?set)
  (if (multifieldp ?new) then
    (if (subset ?new ?set) then 
       (return ?set)
     else (return (create$ ?new ?set)))
  else (if (member$ ?new ?set) then
     	 (return ?set)
   	else (return (insert$ ?set 1 ?new))))
)
;;; example: (add-if-not-member da (du da dum))	--> (du da dum)
;;; 	     (add-if-not-member da (du do dum))	--> (da du do dum)
;;; 	     (add-if-not-member (da dum) (du do dum))	--> (da dum du do dum)
;;; 	     (add-if-not-member (do dum) (du do dum))	--> (du do dum)

;;; (any-common-members ?mv1 ?mv2)
;;; Return: TRUE or FALSE
;; checks whether the two multifield arguments have member(s) in common
(deffunction any-common-member (?mv1 ?mv2)
  (if (neq ?mv1 (create$)) then
    (if (member$ (nth$ 1 ?mv1) ?mv2) then (return TRUE)
     else (return (any-common-member (rest$ ?mv1) ?mv2))))
)

;;; (remove-all ?item ?set)
;;; Return: multi-value
;;; removes all occurrances of the first argument from the multifield variable
(deffunction remove-all (?item ?set)
  (bind ?member (member$ ?item ?set))
  (if ?member then
    (return (remove-all ?item (delete$ ?set ?member ?member)))
   else (return ?set))
)
;;; example: (remove-all da (du da dum da))	--> (du dum)
;;; 	     (remove-all da (du do dum))	--> (du do dum)


;;;----------------------------------------------------------------------------
;;;			CLIPS objects
;;;----------------------------------------------------------------------------

;;; (is-class-thing ?object ?class)
;;; Return: TRUE or FALSE
;;; checks superclasses and direct superclass
(deffunction is-class-thing (?object ?class)
  (or (eq ?class (class ?object)) (superclassp ?class (class ?object)))
)

;;; (inst-name ?value)
;;; Return: instance name
;;; transforms the given value into an instance name
;;; works for symbols, strings, instance addresses, and instance names.
(deffunction inst-name (?value)
  (if (or (symbolp ?value) (stringp ?value)) then
     (return (symbol-to-instance-name (sym-cat ?value))) else
    (if (instance-addressp ?value) then
       (return (instance-name ?value))
   else (return ?value)))
)
;;; example: (inst-name "node-gen47") --> [node-gen47]


;;;----------------------------------------------------------------------------
;;;			writing and formatting
;;;----------------------------------------------------------------------------

;;; mutually recursive functions -- see below for the real definition of this
(deffunction write-values (?style ?ins ?in-length ?file)
)

;;; (value-format ?style ?file ?val)
;;; Return: not to be used
;;; format a value
;;;  handles multifield, numbers, and puts quotes around other things if needed
;;;  multifield values are done separately
(deffunction value-format (?style ?file ?val)
   (if (multifieldp ?val) then 
      (write-values ?style ?val (length ?val) ?file) else 
    (if (numberp ?val) then (format ?file "%g" ?val) else
     (if (eq ?style "noquote") then (format ?file "%s" ?val) else
      (if (eq ?style "single") then (format ?file "'%s'" ?val)
    else (format ?file "\"%s\"" ?val)))))
)
;;; example: (value-format "single" file dudada)	--> 'dudada'
;;;	     (value-format "double" file (du da da))  	--> "du" "da" "da"

;;; (write-values ?style ?ins ?in-length ?file)
;;; Return: not to be used
;;; write the values of a multivalue variable. No commata
(deffunction write-values (?style ?ins ?in-length ?file)
  (bind ?count 1)
  (while (<= ?count ?in-length)  
    (format ?file " ")
    (value-format ?style ?file (nth ?count ?ins))
    (bind ?count (+ ?count 1))
    )
)
;;; example: (write-values "noquote" (du da da) 3  file)  --> du da da

;;; (write-list ?style ?ins ?in-length ?file)
;;; Return: not to be used
;;; write a multivalued variable as a list in square brackets to the file.
;;; No commata
(deffunction write-list (?style ?ins ?in-length ?file)
  (format ?file "[")
  (write-values ?style ?ins ?in-length ?file)
  (format ?file "]")
)
;;; example: (write-values "single" (du da da) 3  file)  --> ['du' 'da' 'da']


;;;----------------------------------------------------------------------------
;;;			string stuff
;;;----------------------------------------------------------------------------

;;; (make-string-value $?val)
;;; Return: string
;;; makes a string from the value. Can handle numbers, symbols, multifield
(deffunction make-string-value ($?val)
  (if (multifieldp $?val) then
     (if (eq 1 (length$ $?val)) then
	(return (str-cat (nth$ 1 $?val)))
      else (return (implode$ $?val)))
   else (return (str-cat $?val)))
)
;;; example: (make-string-value duda)		--> "duda"
;;; 	     (make-string-value 123)		--> "123"
;;; 	     (make-string-value (123 duda 230))		--> "123 duda 230"


;;; (last-char ?string)
;;; Return: string
;;; returns the last character in a string
(deffunction last-char (?string)
  (bind ?length (str-length ?string))
  (return (sub-string ?length ?length ?string))
)
;;; example: (last-char "one two~") 	--> "~"


;;; (strip-lead-whites ?string)
;;; Return: string
;;; recursive
(deffunction strip-lead-whites (?string)
 (bind ?first (sub-string 1 1 ?string))
 (if (or (eq ?first " ") (eq ?first "	")) then
    (return (strip-lead-whites (sub-string 2 (str-length ?string) ?string)))
  else (return ?string))
)
;;; example: (strip-lead-whites "  dida  ")	--> "dida  "

;;; (strip-trail-whites ?string)
;;; Return: string
;;; recursive
(deffunction strip-trail-whites (?string)
 (bind ?last (last-char ?string))
 (if (or (eq ?last " ") (eq ?last "	")) then
    (bind ?length (str-length ?string))
    (return (strip-trail-whites (sub-string 1 (- ?length 1) ?string)))
  else (return ?string))
)
;;; example: (strip-lead-whites "  dida  ")	--> "  dida"

;;; (strip-whites ?string)
;;; Return: string
(deffunction strip-whites (?string)
 (bind ?start-stripped (strip-lead-whites ?string))
 (return (strip-trail-whites ?start-stripped))
)
;;; example: (strip-lead-whites "  dida  ")	--> "dida"

;;; (capitalise-first ?string)
;;; Return: string
(deffunction capitalise-first (?string)
 (bind ?first (sub-string 1 1 ?string))
 (bind ?newfirst (upcase ?first))
 (str-cat ?newfirst (sub-string 2 (str-length ?string) ?string))
)
;;; example: (capitalise-first "duda")	--> "Duda"

