;------------------------------------------------------------------------------
; File:		highlite.clp
; Author:       Jussi Stader
; Date:		6/12/94
; Connections:	HARDY functions; CLIPS windows; HARDY, CLIPS and windows utils
;
; Contents:	CLIPS code for highlighting HARDY nodes and their connections
; Updates:
;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
; Instructions for providing these facilities for a HARDY card type
;------------------------------------------------------------------------------
;
; 1. using the HARDY diagram type manager make two menu options 
;    for the card type's custom menu:
; 	"Highlight Selections" and "De-Highlight Card"
; 
; 2. a. If the diagram card already has an event handler for menu options
;       add two options to the event handler which call
; 
;    b. register a menu event handler
;
; 	  (register-event-handler CustomMenu "DIATYPE" diatype-event-handler)
;
;       where DIATYPE is the HARDY name of the diagram type
; 
; 	and put the following code in a file that you load, e.g. diatype.clp
; 
; (deffunction diatype-event-handler (?card-id ?option)
;   (if (eq ?option "Highlight Selections") then
;      (highlight-selections ?card-id "BLUE") else
;    (if (eq ?option "De-Highlight Card") then
;       (de-highlight-card ?card-id) else
;    TRUE))
; )
; 
; 
; 3. load all relevant files either when starting HARDY (use -clips command 
;    line option) or via the HARDY development window
; 
;    The easiest way to do this is to have a loader.clp file in your working 
;    directory. Here is an example of such a file
;
; (register-event-handler CustomMenu "DIATYPE" diatype-event-handler)
; (load "/home/hardy/hardy/library/clputils/general/clips.clp")
; (load "/home/hardy/hardy/library/clputils/general/hardy.clp")
; (load "/home/hardy/hardy/library/clputils/general/windows.clp")
; (load "/home/hardy/hardy/library/clputils/applics/highlight.clp")
; (load "diatype.clp")
; 
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;	Notes and General Comments
;------------------------------------------------------------------------------
; 
; Currently the colour used for highlighting nodes and arcs is 
; parameterised at the event-handler level - in the above example it is blue.
; This should probably be changed to a choice on the highlight dialogue
; window.
;
; There should be an option that allows users to selected a sub-set of 
; arc/node types to be included in highlighting.
;
; Only the text and the outline or nodes and arcs will change colour, not
; the fill (i.e. arrow heads and the inside of boxes keep their original
; colour).
; 
; De-highlighting will re-assign the colour that the node had before 
; it was first highlighted. This is independent of the colours defined in
; the diagram type. However, if the node changes colour (through other code)
; between being highlighted and de-highlighted, then this new colour is
; ignored when de-highlighting and the original colour is restored.
; 
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------


;------------------------------------------------------------------------------
;	This is where the code starts
;------------------------------------------------------------------------------

(defglobal ?*colour-memory* 		= (create$))
(defglobal ?*hide-memory* 		= (create$))
(defglobal ?*node-highlight-modes* 	= (create$ 1 1 1 1 0 0))
(defglobal ?*arc-highlight-modes* 	= (create$ 0 0 0 0))

(defglobal ?*highlight-mode-window* 	= "")
(defglobal ?*selected-nodes* 		= (create$))
(defglobal ?*selected-arcs* 		= (create$))
(defglobal ?*highlight-card* 		= 0)
(defglobal ?*highlight-colour* 		= "BLUE")


(deffunction draw-all-images (?card ?images)
  (if (neq ?images (create$)) then
    (diagram-image-draw ?card (nth$ 1 ?images))
    (draw-all-images ?card (rest$ ?images)))
)


(deffunction highlight-image (?card ?colour ?image)
  (bind ?*colour-memory* 
 	(add-if-not-member (create$ ?card ?image 
				(diagram-image-get-pen-colour ?card ?image)
    				(diagram-image-get-text-colour ?card ?image))
	      		   ?*colour-memory*))
  (diagram-image-set-pen-colour ?card ?image ?colour)
  (diagram-image-set-text-colour ?card ?image ?colour)
  (diagram-image-draw ?card ?image)
)

(deffunction highlight-node-list (?card ?colour ?images)
  (if (neq ?images (create$)) then
    (highlight-image ?card ?colour  (nth$ 1 ?images))
    (highlight-node-list ?card ?colour (rest$ ?images)))
)
(deffunction highlight-arc-list (?card ?colour ?arc-images)
  (if (neq ?arc-images (create$)) then
    ;;; this should put image to front so that hidden things dont over-lay
    ;;; or should hidden things go to back? (in hide-arcs bit)
    (highlight-image ?card ?colour  (nth$ 1 ?arc-images))
    (highlight-arc-list ?card ?colour (rest$ ?arc-images)))
)

;;; (highlight-lists ?card ?colour ?arcs ?nodes)
;;; Return:
;; highlight the given arc and node images on the card
(deffunction highlight-lists (?card ?colour ?arcs ?nodes)
  (highlight-node-list ?card ?colour ?nodes)
  (highlight-arc-list  ?card ?colour ?arcs)
)


;;; (hide-others-colour ?card ?colour ?nodes ?arcs)
;;; Return:
;; hide all but the given arc and node images on the card
;; does not hide junction nodes
(deffunction hide-others-colour (?card ?colour ?nodes ?arcs)
  (bind ?hidden-nodes (create$))
  (bind ?image (diagram-card-get-first-node-image ?card))
  (while (neq ?image -1) do
    (if (member ?image ?nodes) then TRUE
     else (bind ?hidden (create$ ?hidden-nodes ?image)))
    (bind ?image (diagram-card-get-next-node-image)))
  (bind ?hidden-arcs (create$))
  (bind ?image (diagram-card-get-first-arc-image ?card))
  (while (neq ?image -1) do
    (if (member ?image ?arcs) then TRUE
     else (bind ?hidden-arcs (create$ ?hidden-arcs ?image)))
    (bind ?image (diagram-card-get-next-arc-image)))
  (highlight-lists ?card ?colour ?hidden-arcs ?hidden-nodes)
)

;;; (hide-others-erase ?card ?nodes ?arcs)
;;; Return:
;; hide all but the given arc and node images on the card
;; does not hide junction nodes
(deffunction hide-others-erase (?card ?nodes ?arcs)
  (bind ?hidden (create$))
  (bind ?image (diagram-card-get-first-node-image ?card))
  (while (neq ?image -1) do
    (if (member ?image ?nodes) then TRUE
     else (diagram-image-erase ?card ?image)
          (bind ?*hide-memory* (create$ ?*hide-memory* ?card ?image)))
    (bind ?image (diagram-card-get-next-node-image)))
  (bind ?image (diagram-card-get-first-arc-image ?card))
  (while (neq ?image -1) do
    (if (member ?image ?arcs) then TRUE
     else (diagram-image-erase ?card ?image)
          (bind ?*hide-memory* (create$ ?*hide-memory* ?card ?image)))
    (bind ?image (diagram-card-get-next-arc-image)))
)


;;; (ask-node-highlight-modes ?card)
;;; Return: True if asked False otherwise
;; pop up a window with the yes/no options for what can be highlighted
;;  1: incoming arcs, 2: outgoing arcs,
;;  3: incoming nodes, 4: outgoing nodes, 5: not connected nodes
;;  6: hide rest
;;  the last 3-5 should be dependent, but there's no harm, so no checks done
;; see node-highlight-modes-ok for dealing with selections
(deffunction ask-node-highlight-modes (?card)
  (if (eq ?*highlight-mode-window* "") then 
    (bind ?*highlight-mode-window*
	(make-checkboxes-window  ?card
	 	"Please set the modes for highlighting nodes"
		(create$ "incoming arcs" 
			 "outgoing arcs"
			 "incoming nodes"
			 "outgoing nodes"
			 "nodes not connected"
			 "hide rest")
		?*node-highlight-modes*    ;;; default to previous setting
	 	"node-highlight-ok" "cancel")))
  (window-show ?*highlight-mode-window* 1)
;  (create$ 1 1 1 1 0 1)
)


;;; (get-node-highlight-modes ?card)
;;; Return: mv of length 6 of 0 or 1
;; pop up a window with the yes/no options for what can be highlighted
;;  1: incoming arcs, 2: outgoing arcs,
;;  3: incoming nodes, 4: outgoing nodes, 5: not connected nodes
;;  6: hide rest
;;  the last 3-5 should be dependent, but there's no harm, so no checks done
(deffunction get-node-highlight-modes (?card)
  (create$ 1 1 1 1 0 1)
)
;;; (get-arc-highlight-modes ?card)
;; Return: mv of length 4 of 0 or 1
;; pop up a window with the yes/no options for what can be highlighted
;;  1: from nodes, 2: to nodes, 3: not connected nodes, 4: hide rest
(deffunction get-arc-highlight-modes (?card)
  (create$ 1 1 0 0)
)
(deffunction ask-arc-highlight-modes (?card)
 (format t "cannot work out highlights yet for selected arcs on card %s%n"
	(str-cat ?card))
)


;;; (highlight-node ?card ?node-image ?modes ?colour)
;;; Return: not to be used
;; for node-image, get all incoming arcs and all outgoing arcs separately,
;;  skipping junctions.
;; if either node-option is on, for each node image 
;;  that is not a junction and not the node itself
;; if it meets the options, put it on node-highlight list
;; for both arc options, put highlights on arcs if on, put on arc-highlights
;; for node-highlight list, put highlights on nodes
;; update display for all highlighted things 
(deffunction highlight-node (?card ?node-image ?modes ?colour)
  (bind ?in-arcs  (create$))
  (bind ?out-arcs (create$))
  (bind ?direction "")
  (bind ?highlight-nodes (create$ ?node-image))	; definitely highlight node
  (bind ?highlight-arcs (create$))
  ;;; collect arcs separately
  (bind ?arc-images (get-all-node-image-arcs ?card ?node-image))
  (bind ?count 1)
  (while (<= ?count (length$ ?arc-images)) do
    (bind ?arc-image (nth$ ?count ?arc-images))
    (bind ?other-node (arc-image-get-image-from ?card ?arc-image))
    ;;; put arc into right list, get other node and direction
    (if (eq ?node-image ?other-node) then
      (bind ?out-arcs (append ?out-arcs ?arc-image))
      (bind ?other-node (arc-image-get-image-to ?card ?arc-image))
      (bind ?direction "out")
     else (bind ?in-arcs (append ?in-arcs ?arc-image))
	  (bind ?direction "in"))
    ;;; for junctions, put arc on other side into same direction
    (if (eq 1 (node-image-is-junction ?card ?other-node)) then
      (if (eq ?direction "in") then
	(bind ?in-arcs (append ?in-arcs 
		(get-other-junction-arc ?card ?arc-image ?other-node "from")))
       else (bind ?out-arcs (append ?out-arcs 
		(get-other-junction-arc ?card ?arc-image ?other-node "to")))))
    (bind ?count (+ 1 ?count)))

  ;;; collect nodes to be highlighted
  (if (or (eq 1 (nth$ 3 ?modes)) 		;;; either node option on
	  (eq 1 (nth$ 4 ?modes)) 
	  (eq 1 (nth$ 5 ?modes))) then
    ;;; go through all diagram nodes
    (bind ?other-node (diagram-card-get-first-node-image ?card))
    (while (neq ?other-node -1) do
      (bind ?node-arcs (get-all-node-image-arcs ?card ?other-node))
      (if (or (and (eq 1 (nth$ 3 ?modes))	;; incoming nodes
	           (any-common-member ?node-arcs ?in-arcs))
	      (and (eq 1 (nth$ 4 ?modes))	;; outgoing nodes
	           (any-common-member ?node-arcs ?out-arcs))
	      (and (eq 1 (nth$ 5 ?modes))	;; not connected nodes
	           (not (any-common-member ?node-arcs 
			(append ?in-arcs ?out-arcs))))) 
       then (bind ?highlight-nodes (append ?highlight-nodes ?other-node)))
      (bind ?other-node (diagram-card-get-next-node-image))))

  ;;; highlight the lot
  (if (eq 1 (nth$ 1 ?modes)) then 
    (bind ?highlight-arcs ?in-arcs))
  (if (eq 1 (nth$ 2 ?modes)) then 
    (bind ?highlight-arcs (append ?highlight-arcs ?out-arcs)))
  (if (eq 1 (nth$ 6 ?modes)) then
;     (hide-others-colour ?card "WHITE" ?highlight-nodes ?highlight-arcs)
     (hide-others-erase ?card ?highlight-nodes ?highlight-arcs)
   else TRUE)
  (highlight-lists ?card ?colour ?highlight-arcs ?highlight-nodes)
)

;;; (highlight-arc ?card ?arc-image ?modes)
;;; Return: not to be used
;; for arc-image, get all from-nodes and all to-nodes separately,
;;  skipping junctions.
;; if from-node-option is on, put from-nodes on highlight list
;; if to-node-option is on, put to-nodes on highlight list
;; if not-connected-option is on, go through all node images on the card
;;  if the image is on neither from nor to node lst put it on highlight list
;; for each node on highlight-list, put highlight on
;; update display for all highlighted things 
(deffunction highlight-arc (?card ?arc-image ?modes ?colour)
  (card-set-status-text ?card 
	"cannot work out highlights for selected arcs yet")
)

;;; (highlight-nodes ?card ?modes ?colour ?selected)
;;; Return: not to be used
;; these two functions just iterate over selected images
(deffunction highlight-nodes (?card ?modes ?colour ?selected)
  (if (neq (create$) ?selected) then
    (highlight-node ?card (nth$ 1 ?selected) ?modes ?colour)
    (highlight-nodes ?card ?modes ?colour (rest$ ?selected)))
)
;;; (highlight-arcs ?card ?modes ?colour ?selected)
;;; Return: not to be used
(deffunction highlight-arcs (?card ?modes ?colour ?selected)
  (if (neq (create$) ?selected) then
    (highlight-arc ?card (nth$ 1 ?selected) ?modes ?colour)
    (highlight-arcs ?card ?modes ?colour (rest$ ?selected)))
)


;;; (de-highlight-card ?card)
;;; Return: TRUE
;; remove highlights for card, show hidden images
;; the colour memory has 4 entries per highlighted image: 
;;	card, image, pen-brush, text-brush
;; the hide memory has 2 entries per hidden image: 
;;	card, image
(deffunction de-highlight-card (?card)
  (bind ?count 1)
  (while (<= ?count (length$ ?*colour-memory*)) do
    (if (eq ?card (nth$ ?count ?*colour-memory*)) then
      (bind ?image (nth$ (+ 1 ?count) ?*colour-memory*))
      (diagram-image-set-pen-colour 
		?card ?image (nth$ (+ 2 ?count) ?*colour-memory*))
      (diagram-image-set-text-colour 
		?card ?image (nth$ (+ 3 ?count) ?*colour-memory*))
      (bind ?*colour-memory* 				;;; remove from memory
	(create$ (subseq$ ?*colour-memory* 1 (- ?count 1))
	  (subseq$ ?*colour-memory* (+ 4 ?count) (length$ ?*colour-memory*))))
      (bind ?count (- ?count 4))  	;;; removed 4 entries from memory
      (diagram-image-draw ?card ?image))
    (bind ?count (+ 4 ?count)))	
  ;;; show hidden images
  (bind ?count 1)
  (while (<= ?count (length$ ?*hide-memory*)) do
    (if (eq ?card (nth$ ?count ?*hide-memory*)) then
      (bind ?image (nth$ (+ 1 ?count) ?*hide-memory*))
      (bind ?*hide-memory* 				;;; remove from memory
	(create$ (subseq$ ?*hide-memory* 1 (- ?count 1))
	  (subseq$ ?*hide-memory* (+ 2 ?count) (length$ ?*hide-memory*))))
      (bind ?count (- ?count 2))  	;;; removed 2 entries from memory
      (diagram-image-draw ?card ?image))
    (bind ?count (+ 2 ?count)))	
  TRUE
)

(deffunction node-highlight-ok (?ok-id)
  (window-show ?*highlight-mode-window* 0)
  (bind ?*node-highlight-modes* (check-boxes-get-value ?ok-id))
  (if (neq ?*node-highlight-modes* (create$ 0 0 0 0 0 0)) then
     (highlight-nodes ?*highlight-card* ?*node-highlight-modes* 
		      ?*highlight-colour* ?*selected-nodes*)
     (return TRUE))
)

;;; (highlight-selections ?card ?colour)
;;; Return: not to be used
;;; Collect selected nodes and arcs separately, don't mind types.
;;; If none, complain to user and do nothing. 
;;; any nodes, get node highlight modes; any arcs, get arc highlight modes
;;; if either modes empty, ignore those selections
;;; call highlight-node/arc for each selected thing, passing modes as arg
(deffunction highlight-selections (?card ?colour)
  (bind ?*selected-nodes* (get-selected-type-image ?card "node" "" "all"))
  (bind ?*selected-arcs* (get-selected-type-image ?card "arc" "" "all"))
  (if (neq ?*selected-nodes* (create$)) then
     (ask-node-highlight-modes ?card))
  (if (neq ?*selected-arcs* (create$)) then
    (card-set-status-text ?card 
	"cannot work out highlights for selected arcs yet"))
;     (ask-arc-highlight-modes ?card))
  (if (and (eq ?*selected-nodes* (create$)) (eq ?*selected-arcs* (create$)))
   then (no-selection-message ?card)	;;; nothing selected, complain
   else (bind ?*highlight-card* ?card) 	;;; remember card for ok button code
  (bind ?*highlight-colour* ?colour))  ;;; remember colour
  (return 1)
)
