;;;;;;;;;;;;;;;;;;;;;;; EVENT HANDLER FUNCTIONS ;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; File:   event_handlers.clp
;;; Author: Ian Harrison
;;; Date:   7/794
;;;
;;; Description: This file contains the event handler functions that
;;;              have been declared in the batch file loader.clp, together
;;;              with some auxiliary functions.
;;;
;;; Functions: newNode             (event: CreateNodeImage)     
;;;            exitHardy           (event: Exit)
;;;            saveNetwork
;;;            doAnalysis
;;;            customMenu          (event: CustomMenu)
;;;            setTitle            (event: CreateCard)
;;;            nodeLeftClick       (event: NodeLeftClick)
;;;            nodeRightClick      (event: NodeRightClick)


;;; Function: newNode
;;; Arguments: a diagram card id, an image id and a node-type
;;; Description: It is the callback function of the event 
;;;              "CreateNodeImage".

(deffunction newNode (?card ?image ?type)
    (bind ?object (diagram-image-get-object ?card ?image))
    (bind ?name (get-text-from-user
		   "Enter name of activity." "" 1 (card-get-frame ?card)))
    (if (= 0 (str-compare ?name "")) then
	(message-box "You must enter a name for the activity"
		     OK
		     1 (card-get-frame ?card))
	(return 0)
     else
       (bind ?duration 	(get-text-from-user
			   "Enter duration of activity." "" 1 (card-get-frame ?card)))
       (if (not (numberp (string-to-float ?duration))) then
	    (message-box "You must enter a number for the duration"
		     OK
		     1 (card-get-frame ?card))
	    (return 0)
	else
          (diagram-object-set-string-attribute 
		?card ?object "name" ?name)
          (diagram-object-set-string-attribute 
		?card ?object "duration" ?duration)
	  (diagram-object-format-text ?card ?object)
	)
    )
)

;;; Function saveNetwork
;;; Argument: a diagram card id
;;; Description: First the diagram is saved. In case that the user has 
;;;	         not given a diagram file name yet the title of the 
;;;              root card is chosen as the file name. 

(deffunction saveNetwork (?card)
    (bind ?root (diagram-card-find-root ?card))
    (bind ?filename (card-get-string-attribute ?root "filename"))
    (if (= 0 (str-compare ?filename "")) then
       (bind ?filename (get-text-from-user
		   	 "Enter name of file for diagram." 
			 "" 1 (card-get-frame ?card)))
       (if (not (= 0 (str-compare ?filename ""))) then
          (bind ?indexname (str-cat ?filename ".ind"))
          (bind ?filename (str-cat ?filename ".dia"))
          (card-set-string-attribute ?root "filename" ?filename)
          (diagram-card-save-file ?root ?filename)
          (save-index ?indexname)
	else
          (message-box "You must enter a file name"
		        OK
		        1 (card-get-frame ?card))
          (return 0)))
)

;;; Function doAnalysis
;;; Argument: a diagram card id
;;; Description: The critical path analysis for diagram, starting at
;;;              the selected node is done.

(deffunction doAnalysis (?card)
    (if (and (= 1 (diagram-image-selected ?card  ?*selected-image-id*))
             (= 1 (validStartNode ?card ?*selected-image-id*))) then
	(criticalPath ?card ?*selected-image-id*)
;        (diagram-image-select ?card ?*selected-image-id* 0) ; deselect it
;        (bind ?*selected-image-id* 0)
        (redraw-diagram ?card)
     else
         (message-box "You must select a valid start node for the analysis"
		     OK
		     1 (card-get-frame ?card))
	 (return 1)
     )
)

;;; Function: customMenu
;;; Argument: a diagram card id and a menu option of the custom menu
;;; Description: Callback function of the custom menu.

(deffunction customMenu (?card ?string) 
    (if (= 0 (str-compare ?string "Save network")) then 
	(saveNetwork ?card))
    (if (= 0 (str-compare ?string "Critical path")) then
        (doAnalysis ?card)
    )
)

;;; Function: setTitle
;;; Argument: a diagram card id
;;; Description: This is the callback function of "CreateCard". If
;;;              the user creates a new diagram card then he is asked
;;;              to give a unique title to it.

(deffunction setTitle (?card)
    (bind ?title (card-get-string-attribute ?card "title"))
    (bind ?title (get-text-from-user
    		     	"Please enter a title for the new card." 
			?title 
			1 (card-get-frame ?card)))
    (if (= 0 (str-compare ?title "")) then
	(bind ?title "No title")
	(card-set-string-attribute ?card "title" ?title)
     else
        (card-set-string-attribute ?card "title" ?title))
)

(deffunction initAttributeFrame (?card ?node-object))

;;; Function: nodeLeftClick
;;; Arguments: a diagram card id, a node image id, x and y position, 
;;;            shift pressed and control pressed
;;; Description: When node image is clicked on to select it, the
;;;              global ?*selected-image-id* gets the id of the
;;;              selected image.

(deffunction nodeLeftClick (?card ?node-image ?x ?y ?shift ?control)
     (if (and (= ?shift 0) (= ?control 0)) then 
        (if (= 1 (diagram-image-selected ?card ?*selected-image-id*)) then
           (diagram-image-select ?card 
				 ?*selected-image-id* 0) ; deselect selection
	   (bind ?*selected-image-id* ?node-image) ; set selection to node
           (diagram-image-select ?card ?node-image 1) ; select node
           (diagram-card-redraw ?card)
         else
           (bind ?*selected-image-id* ?node-image)
           (diagram-image-select ?card ?node-image 1) ; select node
           (diagram-card-redraw ?card)
        )
     )
     (if (and (= ?shift 0) (= ?control 1)) then
        (bind ?node-object (diagram-image-get-object ?card ?node-image))
	(initAttributeFrame ?card ?node-object)
        (return 0)
      else
        (return 0))
)

;;; Function: nodeRightClick
;;; Arguments: a card id, a node image id, x and y position, shift
;;;            pressed and control pressed
;;; Description: Overides the default behaviour of right-clicking
;;;              on a node --- i.e. the user cannot edit the attributes
;;;              for a node.

(deffunction nodeRightClick (?card ?image ?x ?y ?shift ?control) 
    (return 0)
)










