;;; Function getChildren
;;; Argument: card id and image id of a node in the graph
;;; Description: Returns a multifield variable of image id's of those
;;;              children nodes connected to the node.

(deffunction getChildren (?card ?node-image)
    (bind ?node-object (diagram-image-get-object ?card ?node-image))
    (bind ?arc-object 
	(node-object-get-first-arc-object ?card ?node-object))
    (bind ?arc-image (diagram-object-get-first-image ?card ?arc-object))
;    (printout t "arc image = " ?arc-image crlf)
    (bind $?children (create$))
    (while (<> ?arc-image -1) do
	(bind ?connected-image (arc-image-get-image-to ?card ?arc-image))
        (bind ?connected-object 
		(diagram-image-get-object ?card ?connected-image))
        (if (neq ?connected-object ?node-object) then
	   (bind $?children (create$ $?children ?connected-image)))
	(bind ?next-arc-object (node-object-get-next-arc-object))
;        (printout t "while loop next arc object = " ?next-arc-object crlf)
        (if (= ?next-arc-object -1) then
	   (bind ?arc-image -1)
         else       
           (bind ?arc-image 
		(diagram-object-get-first-image ?card ?next-arc-object))
        )
;        (printout t "while loop next arc image = " ?arc-image crlf)
    )
    (return $?children)
)

;;; Function initialiseNodeList
;;; Argument: card id and image id of the root node of a graph
;;; Description: Returns a multifield variable of image id's of those
;;;              nodes connected to the root node.

(deffunction initialiseNodeList (?card ?root-node-image)
    (bind $?children (getChildren ?card ?root-node-image))
    (return (create$ ?root-node-image $?children))
)


;;; Function: getParents
;;; Argument: card id and image id of a node in the graph
;;; Description: Returns a multifield variable of image id's of those
;;;              parent nodes connected to the node.

(deffunction getParents (?card ?node-image)
    (bind ?node-object (diagram-image-get-object ?card ?node-image))
    (bind ?arc-object 
	(node-object-get-first-arc-object ?card ?node-object))
    (bind ?arc-image (diagram-object-get-first-image ?card ?arc-object))
;    (printout t "arc image = " ?arc-image crlf)
    (bind $?parents (create$))
    (while (<> ?arc-image -1) do
	(bind ?connected-image (arc-image-get-image-from ?card ?arc-image))
        (bind ?connected-object 
		(diagram-image-get-object ?card ?connected-image))
        (if (neq ?connected-object ?node-object) then
	   (bind $?parents (create$ $?parents ?connected-image)))
	(bind ?next-arc-object (node-object-get-next-arc-object))
;        (printout t "while loop next arc object = " ?next-arc-object crlf)
        (if (= ?next-arc-object -1) then
	   (bind ?arc-image -1)
         else       
           (bind ?arc-image 
		(diagram-object-get-first-image ?card ?next-arc-object))
        )
;        (printout t "while loop next arc image = " ?arc-image crlf)
    )
    (return $?parents)
)

;;; Function validStartNode
;;; Argument: card id and image id of a node in the graph
;;; Description: If node is a valid start node for analysis return 1
;;;              otherwise return 0. A valid start node is one with
;;;              no parents, but does have children

(deffunction validStartNode (?card ?selected-image-id)
   (bind $?parents (getParents ?card ?selected-image-id))
   (bind $?children (getChildren ?card ?selected-image-id))
   (if (and (= (length $?parents) 0)
            (> (length $?children) 0)) then
      (return 1)
    else
      (return 0)))

;;; Function processedNode
;;; Argument: card id and image id of a node in the graph
;;; Description: If node has been processed in a forward direction
;;;              return 1, if it has been processed in both forward 
;;;              and backward direction return 2, else return 0.

(deffunction processedNode (?card ?node-image)
    (bind ?node-object (diagram-image-get-object ?card ?node-image))
    (bind ?earliest-finish 
	    (diagram-object-get-string-attribute ?card ?node-object 
				"earliest finish"))
    (bind ?latest-start 
	    (diagram-object-get-string-attribute ?card ?node-object 
				"latest start"))
    (if (and (= 0 (str-compare ?latest-start ""))
	     (= 0 (str-compare ?earliest-finish ""))) then	
      (return 0))
    (if (<> 0 (str-compare ?latest-start "")) then
	(return 2))
    (if (<> 0 (str-compare ?earliest-finish "")) then
	(return 1))
)
   
;;; Function unprocessedParents
;;; Argument: card id and image id of a node in the graph
;;; Description: Returns 1 if any of the parent nodes are unprocessed
;;;              in a forward direction and returns 0 if all parents 
;;;              are processed.

(deffunction unprocessedParents (?card ?node-image)
   (bind $?parents (getParents ?card ?node-image))
   (while (> (length $?parents) 0) do
       (bind ?parent-image (nth$ 1 $?parents))
       (bind ?processed-state (processedNode ?card ?parent-image))
       (if (= 0 ?processed-state) then
	 (return 1)
        else 
         (bind $?parents (delete$ $?parents 1 1))
       )
    )
    (return 0))

;;; Function highestConnectedEF
;;; Argument: card id and image id of a node in the graph
;;; Description: Returns the attribute string that is the highest number
;;;              of all the node's earliest finish.

(deffunction highestConnectedEF (?card ?node-image)
   (bind $?parents (getParents ?card ?node-image))
   (bind ?maximum 0)
   (while (> (length $?parents) 0) do
       (bind ?parent-image (nth$ 1 $?parents))
       (bind ?parent-object (diagram-image-get-object ?card ?parent-image))
       (bind ?earliest-finish 
	    (diagram-object-get-string-attribute ?card ?parent-object 
				"earliest finish"))
       (if (> (string-to-float ?earliest-finish) ?maximum) then
	  (bind ?maximum (string-to-float ?earliest-finish)))
       (bind $?parents (delete$ $?parents 1 1))
   )
   (return (str-cat "" ?maximum))
)

;;; Function processNode
;;; Argument: card id and image id of a node in the graph
;;; Description: If node has been processed jump out of processing.
;;;              If node has any unprocessed parents add these to
;;;              end of process list and jump out of processing.
;;;              Otherwise process the node and then append children
;;;              of node to the process-list

(deffunction processNode (?card ?node-image)
   (if (> 0 (processedNode ?card ?node-image)) then
     (return 1))
   (if (= 1 (unprocessedParents ?card ?node-image)) then
	(bind ?*process-list*
		(create$ ?*process-list* (getParents ?card ?node-image)))
      (return 1))
   (if (and (= 0 (unprocessedParents ?card ?node-image))
            (= 0 (processedNode ?card ?node-image))) then
     (bind ?earliest-start (highestConnectedEF ?card ?node-image))
;     (printout t "Highest earliest start = " 
;                 ?earliest-start crlf)
     (bind ?node-object (diagram-image-get-object ?card ?node-image))
     (bind ?duration 
             (string-to-float (diagram-object-get-string-attribute
		                ?card ?node-object "duration")))
;     (printout t "Duration = " ?duration crlf)
     (diagram-object-set-string-attribute 
		     ?card ?node-object "earliest start" ?earliest-start)
     (bind ?earliest-finish (+ ?duration
			          (string-to-float ?earliest-start)))
;     (printout t "Earliest finish = " ?earliest-finish crlf)
     (diagram-object-set-string-attribute 
		?card ?node-object "earliest finish" 
				    (str-cat "" ?earliest-finish))
;     (printout t "Stored EF = "
;        (diagram-object-get-string-attribute 
;		?card ?node-object "earliest finish" ) crlf)
     (diagram-object-format-text ?card ?node-object)
     (bind ?*process-list* 
	        (create$ ?*process-list* (getChildren ?card ?node-image)))
     (return 1)
   )
)

;;; Function unprocessedChildren
;;; Argument: card id and image id of a node in the graph
;;; Description: Returns 1 if any of the children nodes are unprocessed
;;;              in a reverse direction and returns 0 if all children
;;;              are processed.

(deffunction unprocessedChildren (?card ?node-image)
   (bind $?children (getChildren ?card ?node-image))
   (while (> (length $?children) 0) do
       (bind ?child-image (nth$ 1 $?children))
       (bind ?processed-state (processedNode ?card ?child-image))
       (if (or (= 0 ?processed-state)
               (= 1 ?processed-state)) then
	 (return 1)
        else
         (bind $?children (delete$ $?children 1 1))
       )
    )
    (return 0))

;;; Function lowestConnectedLS
;;; Argument: card id and image id of a node in the graph
;;; Description: Returns the attribute string that is the lowest number
;;;              of all the node's latest start.

(deffunction lowestConnectedLS (?card ?node-image)
   (bind $?children (getChildren ?card ?node-image))
   (if (= 0 (length $?children)) then ;; first node take
                                      ;; earliest finish as minimum
      (bind ?node-object (diagram-image-get-object 
                                 ?card ?node-image))
      (bind ?minimum  (string-to-float 
                        (diagram-object-get-string-attribute 
                                ?card ?node-object 
				"earliest finish")))
;      (printout t "Root node minumum = " ?minimum crlf)
    else
      (bind ?first-child (diagram-image-get-object ?card (nth$ 1 $?children)))
      (bind ?minimum  (string-to-float 
                         (diagram-object-get-string-attribute ?card 
                                ?first-child "latest start")))
      (while (> (length $?children) 0) do
         (bind ?child-image (nth$ 1 $?children))
         (bind ?child-object (diagram-image-get-object ?card ?child-image))
         (bind ?latest-start 
	           (diagram-object-get-string-attribute ?card ?child-object 
				                        "latest start"))
         (if (< (string-to-float ?latest-start) ?minimum) then
	    (bind ?minimum (string-to-float ?latest-start)))
            (bind $?children (delete$ $?children 1 1))
       )
   )
;   (printout t "Minumum = " ?minimum crlf)
   (return (str-cat "" ?minimum))
)

;;; Function: reverseProcessNode
;;; Argument: card id and image id of a node in the graph
;;; Description: If node has been processed jump out of processing.
;;;              If node has any unprocessed children add these to
;;;              end of process list and jump out of processing.
;;;              Otherwise process the node and then append parents
;;;              of node to the process-list


(deffunction reverseProcessNode (?card ?node-image)
   (if (> 0 (processedNode ?card ?node-image)) then
     (return 1))
   (if (= 1 (unprocessedChildren ?card ?node-image)) then
	(bind ?*process-list*
		(create$ ?*process-list* (getChildren ?card ?node-image)))
      (return 1))
   (if (and (= 0 (unprocessedChildren ?card ?node-image))
            (< (processedNode ?card ?node-image) 2)) then
     (bind ?latest-finish (lowestConnectedLS ?card ?node-image))
;     (printout t "Highest earliest start = " 
;                 ?earliest-start crlf)
     (bind ?node-object (diagram-image-get-object ?card ?node-image))
     (bind ?duration 
             (string-to-float (diagram-object-get-string-attribute
		                ?card ?node-object "duration")))
;     (printout t "Duration = " ?duration crlf)
     (diagram-object-set-string-attribute 
		     ?card ?node-object "latest finish" ?latest-finish)
     (bind ?latest-start (- (string-to-float ?latest-finish)
                             ?duration))
;     (printout t "Earliest finish = " ?earliest-finish crlf)
     (diagram-object-set-string-attribute 
		?card ?node-object "latest start" 
				    (str-cat "" ?latest-start))
;     (printout t "Stored EF = "
;        (diagram-object-get-string-attribute 
;		?card ?node-object "earliest finish" ) crlf)
     (diagram-object-format-text ?card ?node-object)
     (bind ?*process-list* 
	        (create$ ?*process-list* (getParents ?card ?node-image)))
     (return 1)
   )
)

;;; Function: reverseProcess
;;; Argument: card id and image id of the last node of a graph
;;; Description: Calculates the latest start and latest finish 
;;;              for the nodes in the graph

(deffunction reverseProcess (?card ?root-node-image)
    (bind ?*process-list* (create$ ?root-node-image 
                                   (getParents ?card ?root-node-image)))
    (while (> (length ?*process-list*) 0) do
	(bind ?node-image (nth$ 1 ?*process-list*))
	(reverseProcessNode ?card ?node-image)
	(bind ?*process-list* (delete$ ?*process-list* 1 1))
    )
)

;;; Function: resetAttributes
;;; Argument: card id and node object id
;;; Description: Resets the earliest start, earliest finish, latest
;;;              start and latest finish attributes of all nodes.

(deffunction resetAttributes (?card ?node-object)
   (diagram-object-set-string-attribute 
        ?card ?node-object "earliest start" "")    
   (diagram-object-set-string-attribute 
        ?card ?node-object "earliest finish" "")    
   (diagram-object-set-string-attribute 
        ?card ?node-object "latest start" "")    
   (diagram-object-set-string-attribute 
        ?card ?node-object "latest finish" "")
)

;;; Function: initialiseDiagram
;;; Argument: card id
;;; Description: Resets the earliest start, earliest finish, latest
;;;              start and latest finish attributes of all nodes.

(deffunction initialiseDiagram (?card)
    (bind ?node-object (get-first-card-node ?card))
     (while (> ?node-object 0) do
       (resetAttributes ?card ?node-object)
       (bind ?node-object (get-next-card-node))
     )
    (return 1))

;;; Function: arcToCriticalNode
;;; Argument: card id, image id of a node in the graph on critical path,
;;;           earliest finish and latest finish
;;; Description: Returns image id of arc to next node on critical path

(deffunction arcToCriticalNode (?card ?node-image 
				?earliest-finish ?latest-finish)
  (bind ?node-object (diagram-image-get-object ?card ?node-image))
  (bind ?arc-object 
	(node-object-get-first-arc-object ?card ?node-object)) 
  (bind ?arc-image (diagram-object-get-first-image ?card ?arc-object))
  (bind ?cp-arc-image -1)
  (while (<> ?arc-image -1) do
     (bind ?connected-image (arc-image-get-image-to ?card ?arc-image))
     (bind ?connected-object 
		(diagram-image-get-object ?card ?connected-image))
     (bind ?connected-earliest-start 
	     (string-to-float (diagram-object-get-string-attribute 
					?card ?connected-object 
				        "earliest start")))
     (bind ?connected-latest-start 
	     (string-to-float (diagram-object-get-string-attribute 
					?card ?connected-object 
					"latest start")))
     (if (and (= ?earliest-finish ?connected-earliest-start)
              (= ?latest-finish ?connected-latest-start)) then
     	(bind ?cp-arc-image ?arc-image))
     (bind ?next-arc-object (node-object-get-next-arc-object))
     (if (= ?next-arc-object -1) then
         (bind ?arc-image -1)
      else       
         (bind ?arc-image 
	        (diagram-object-get-first-image ?card ?next-arc-object))
     )
  )
  (return ?cp-arc-image)
)

(deffunction markAll (?card ?node-image ?earliest-finish ?latest-finish)
   (bind ?cp-arc-image (arcToCriticalNode ?card ?node-image
                                          ?earliest-finish ?latest-finish))
   (if (> ?cp-arc-image 0) then
     (diagram-image-add-annotation ?card ?cp-arc-image 
                                   "Hollow circle arrowhead"
                                   "Middle")
     (diagram-image-set-pen-colour ?card ?cp-arc-image "RED")
     (bind ?connected-image (arc-image-get-image-to ?card ?cp-arc-image))
     (bind ?connected-object 
		(diagram-image-get-object ?card ?connected-image))
     (bind ?connected-earliest-finish 
	      (string-to-float 
		  (diagram-object-get-string-attribute ?card 
				?connected-object "earliest finish")))
     (bind ?connected-latest-finish 
	     (string-to-float
		  (diagram-object-get-string-attribute ?card 
			          ?connected-object "latest finish")))
     (markAll ?card ?connected-image ?connected-earliest-finish
                                     ?connected-latest-finish)
   else
     (return 1))
)

;;; Function: markCriticalPath
;;; Argument: card id and image id of the root node of the graph
;;; Description: Marks the critical path on graph

(deffunction markCriticalPath (?card ?node-image)
   (bind ?node-object (diagram-image-get-object ?card ?node-image))
   (bind ?earliest-finish 
                 (string-to-float (diagram-object-get-string-attribute
          	                     ?card ?node-object "earliest finish")))
   (bind ?latest-finish 
                 (string-to-float (diagram-object-get-string-attribute
		                  ?card ?node-object "latest finish")))
   (markAll ?card ?node-image ?earliest-finish ?latest-finish)
)

;;; Function: removeOldCriticalPath
;;; Argument: card id
;;; Description: Loops through all arcs in the card and removes the
;;;              "Hollow circle arrowhead" annotation.

(deffunction removeOldCriticalPath (?card)
   (bind ?arc-image (diagram-card-get-first-arc-image ?card))
   (while (<> ?arc-image -1) do
      (diagram-image-set-pen-colour ?card ?arc-image "BLACK")
      (bind ?annotation 
		(diagram-image-get-first-annotation ?card ?arc-image))
      (bind ?annotation-name
                (diagram-image-annotation-get-name
                      ?card ?arc-image ?annotation))
      (while (<> ?annotation -1) do
         (if (eq ?annotation-name "Hollow circle arrowhead") then
            (diagram-image-delete-annotation ?card ?arc-image ?annotation))
         (bind ?annotation
                (diagram-image-get-next-annotation))
         (bind ?annotation-name
                  (diagram-image-annotation-get-name
                      ?card ?arc-image ?annotation))
      )
      (bind ?arc-image (diagram-card-get-next-arc-image))
   )
)
   
;;; Function: criticalPath
;;; Argument: card id and image id of the root node of a graph
;;; Description: Calculates the critical path for the graph and 
;;;              as a side effect, puts all the time values into the
;;;              appropriate attributes of all the node. Returns 1.


(deffunction criticalPath (?card ?root-node-image)
;    (printout t "got into function criticalPath" crlf)
    (initialiseDiagram ?card)
    (bind ?*process-list* (initialiseNodeList ?card ?root-node-image))
;    (printout t "got back from initialiseNodeList" crlf)
;    (printout t "Initial list of node image id's = " ?*process-list* crlf)
    (while (> (length ?*process-list*) 1) do
	(bind ?node-image (nth$ 1 ?*process-list*))
	(processNode ?card ?node-image)
	(bind ?*process-list* (delete$ ?*process-list* 1 1))
    )
    (bind ?node-image (nth$ 1 ?*process-list*))
    (processNode ?card ?node-image)
    (reverseProcess ?card ?node-image)
    (removeOldCriticalPath ?card)
    (diagram-card-clear-canvas ?card)
    (diagram-card-redraw ?card)
    (markCriticalPath ?card ?root-node-image)
;    (printout t "processed list of node image id's = " ?*process-list* crlf)
    (return 1))

    
