/*
 * @(#)FtpModule.java	1.0 98/1/5
 *
 * This module started its life as an example that was made available 
 * by Joseph Szabo (see: http://remus.rutgers.edu/~jszabo/myjava).
 * Several parts have been modified and updated to comply with Applet
 * security.
 *
 * Currently, Cpe uses getFile and sendFile... 
 */

import java.io.*;
import java.net.*;
import java.util.*;
import CpeProcess;
import CpeProcessPanel;
import com.sun.java.swing.*;
import java.beans.*;
import java.awt.*;
import com.sun.java.swing.tree.*;

/**
 * This object handles the interaction between the GUI User Interface and the
 * ftp protocol's control and data ports. All logic necessary to carry out
 * an ftp command is contained in this class.
 *
 * @version 1.0 98/1/5
 * @author Steve Polyak 
 */
public class FtpModule {
	
  /**
   * These values are used to represent the first digit in a
   * three digit response from the ftpd running on the remote
   * host.
   */
  private static final int POSITIVE_PRELIMINARY_REPLY  = 1;	 
  private static final int POSITIVE_COMPLETION_REPLY   = 2;	 
  private static final int POSITIVE_INTERMEDIATE_REPLY = 3;	 
  private static final int TRANSIENT_NEGATIVE_REPLY    = 4;	
  private static final int PERMANENT_NEGATIVE_REPLY    = 5;	 
  


  /** 
   * These vectors hold the id's during the save to produce the
   * SORT lines at the end.
   */

  Vector cpo_plan;
  Vector cpo_process;
  Vector cpo_activity_spec;
  Vector cpo_action;
  Vector cpo_start;
  Vector cpo_finish;
  Vector cpo_begin;
  Vector cpo_end;
  Vector cpo_timepoint;
  Vector cpo_ordering_c;
  Vector cpo_input_c;
  Vector cpo_output_c;
  Vector cpo_resource_c;
  Vector cpo_include_c;
  Vector cpo_annot_c;
  Vector cpo_unit;
  Vector cpo_var;
  
  private boolean tfExport = false;

  /**
   * This class tracks the state of the connection.
   * These static constants represent the possible states.
   */
  private static final int IDLE      = 1;
  private static final int CONNECTED = 2;
  
  private Integer m_port = null;
  
  /**
   * Used internally to keep track of the state of the engine 
   * Valid states are: 
   *       IDLE      - Waiting for some action.
   *       CONNECTED - Control port connected to a server.
   *
   */
  private int State = IDLE;
  
  private OutputStream m_outdataport = null;

  /**
   * Control Port Variables. These variable are used to represent
   * the control socket to the remote ftpd.
   */
  private Socket          controlSocket       = null;
  private DataInputStream controlInputStream  = null;
  private PrintStream     controlOutputStream = null;
  private ByteArrayOutputStream  tfOutputStream = null;
  
  public CpeProcessPanel m_frame = null;
  public JTree tree = null;

  private int m_frameCount = 0;
  /**
   * Constructor
   *
   * @param parent The object that created this interface must
   *               implement the TextDisplayable interface and
   *               pass in a reference of itself at time of 
   *               creation. This is used to feed text back to
   *               a scrolling window for user feedback.
   */
  public FtpModule(JTree tree) {
    //m_frame = frame;
    this.tree = tree;
  }
  
  
  /**
   * Sends readable text back to the user's display for
   * feedback when the debug flag is on.
   *
   * @param text  Text to print into the user window.
   */
  private void printText(String text) {

    if (Cpe.sharedInstance().debugFlag == true) {
      System.out.println(text);
    }
  }
  
  /** 
   * Used to connect to the remote machine and initiate comunication.
   * This method only takes care of establishing the control socket.
   * No atempt is made by this method to login.
   *
   * @param hostName Name of the host to connect to.
   *
   * @return Returns true on success.
   */
  public boolean connect (String hostName) {
    
    if(State == CONNECTED) {
      // Disconnect the current connection first
      this.disconnect();
    }
    
    try {
      controlSocket       = new Socket(hostName, 21);  // ftp port
      controlInputStream  = new 
	DataInputStream(controlSocket.getInputStream());
      controlOutputStream = new 
	PrintStream(controlSocket.getOutputStream());
      State = CONNECTED;
    } catch (UnknownHostException e) {
      printText("ERR Don't know about host: " + hostName);
      return(false);
    } catch (IOException e) {
      printText("ERR Couldn't get I/O for the connection to: " + hostName);
      return(false);
    }
    
    // Initial intro response
    checkResponse();
    return(true);
  } 
  
  /** 
   * Causes the control port to be disconnected and closed.
   * This method also cleans up any data port connection that
   * may be active. 
   *
   * @return Returns true on success.
   */
  public boolean disconnect() {
    
    if (controlSocket != null &&
	controlOutputStream != null &&
	controlInputStream != null) {
      try {
	
	// Send the quit command
	String tempCmd = new String("QUIT\r");
	printText(tempCmd);
	controlOutputStream.println(tempCmd);
	int result = checkResponse();
	
	controlOutputStream.close();
	controlInputStream.close();
	controlSocket.close();
	State = IDLE;
      } catch (IOException e) {
	printText("ERR I/O failed while closing connection!");
	return(false);
      }
    }
    return(true);
  } 
	
  /**
   * This method is used to authenticate the user.
   *
   * @param userName Name of the user on the remote host.
   * @param passWord Password to send to the remote host.
   *
   * @return Returns true on success.
   */
  public boolean login(String userName, String passWord) {
    
    int result;
    String tempCmd;
    
    if(State == CONNECTED) {
      
      tempCmd = new String("USER " + userName + "\r");
      printText(tempCmd);
      controlOutputStream.println(tempCmd);
      result = checkResponse();
      
      if (result == POSITIVE_INTERMEDIATE_REPLY) {
	
	tempCmd = new String("PASS " + passWord + "\r");
	printText("PASS xxxxxx");
	controlOutputStream.println(tempCmd);
	result = checkResponse();
	
	if(result == POSITIVE_COMPLETION_REPLY) {
	  return(true);
	}
	
      }
    }
    return(false);
  }
  
  /**
   * Get current directory on the remote machine.
   *
   * @return Name of the current directory on the remote machine.
   */
  public String getCurrentDir() {
    
    int i;
    int result;
    String tempCmd;
    String inLine;
    String word;
    String workingDir;
    StringTokenizer st;
    
    workingDir = "";
    if(State == CONNECTED) {
      
      printText("PWD");
      controlOutputStream.println("PWD\r");
      
      try {
	do {
	  inLine = controlInputStream.readLine();
	  printText(inLine);
	} while(!(Character.isDigit(inLine.charAt(0)) &&
		  Character.isDigit(inLine.charAt(1)) &&
		  Character.isDigit(inLine.charAt(2)) &&
		  inLine.charAt(3) == ' '));
      } catch (IOException e) {
	printText("ERR Error getting reply from controlport!");
	return("");
      }
      
      // Now parse the line returned and grab the second token which
      // should be the name of the current working directory on the
      // remote host.
      st = new StringTokenizer( inLine );
      i = 0;
      while( st.hasMoreTokens() && workingDir.equals("") ) {
	word = st.nextToken();
	i++;
	if(i == 2) {
	  workingDir = new String( word.substring(1,word.length()-1));
	  break;
	}
      }
    }
    return(workingDir);
  }
  
  /**
   * Change directory on the remote host.
   *
   * @param newDir New directory on the remote host into which
   *               the current directory should be changed.
   *
   * @return Returns true on success.
   */
  public boolean chdir(String newDir) {
    
    int result;
    String tempCmd;
    
    if(State == CONNECTED) {
      tempCmd = new String("CWD " + newDir + "\r");
      printText(tempCmd);
      controlOutputStream.println(tempCmd);
      result = checkResponse();
			
      if(result == POSITIVE_COMPLETION_REPLY) {
	return(true);
      }
    }
    return(false);
  }
  
  
  /**
   * This method is used to set the type of data transfer that 
   * is to be performed. The two supported methods are "ASCII"
   * and "BINARY".
   *
   * @param mode  This should be one of two values "ASCII" 
   *              or "BINARY". The default is "BINARY" in 
   *              the case that a bad parameter is passed in.
   */
  public void setType(String mode) {
    
    if(State == CONNECTED) {
      if(mode.equals("ASCII")) {
	controlOutputStream.println("TYPE A\r");
	printText("TYPE A");
	checkResponse();
      } else {
	// Everything else should be "BINARY"
	controlOutputStream.println("TYPE I\r");
	printText("TYPE I");
	checkResponse();
      }
    }
  }
  
  
  /**
   * Send port command to far end and notify it of the fact that
   * this end is listening on a socket for a connection.
   *
   * @param servSock  This is the listening socket on which this
   *                  program is listening for a connection.
   */
  public void setPort(ServerSocket servSock) {
    String str;
    int result;
    
    if(State == CONNECTED) {
      str = new String("PORT " + calcPortCmdParameter(servSock) + "\r");
      printText(str);
      controlOutputStream.println(str);
      result = checkResponse();
    }
  }
  
  
  /**
   * Delete a file on remote host.
   *
   * @param fileName  This is the fileName of the file to delete
   *                  on the remote server.
   */
  public void delete(String fileName) {
    String str;
    int result;
    
    if(State == CONNECTED) {
      str = new String("DELE " + fileName + "\r");
      printText(str);
      controlOutputStream.println(str);
      result = checkResponse();
    }
  }
  
  
  /**
   * Create a new directory on the remote server.
   *
   * @param dirName  This is the directory name to be used for the new 
   *                 directory on the remote server.
   */
  public void mkdir(String dirName) {
    
    if(State == CONNECTED) {
      String str = new String("MKD " + dirName + "\r");
      printText(str);
      controlOutputStream.println(str);
      checkResponse();
    }
  }
  
  
  /**
   * Remove a directory from the remote server.
   *
   * @param dirName  This is the directory name to be removed 
   *                 from the remote server.
   */
  public void rmdir(String dirName) {
    
    if(State == CONNECTED) {
      String str = new String("RMD " + dirName + "\r");
      printText(str);
      controlOutputStream.println(str);
      checkResponse();
    }
  }
  
  
  /**
   * Rename a file on remote server.
   *
   * @param oldName The current name of the file.
   * @param newName The new name of the file.
   */
  public void rename(String oldName, String newName) {
    
    if(State == CONNECTED) {
      String tempCmd = new String("RNFR " + oldName + "\r");
      printText(tempCmd);
      controlOutputStream.println(tempCmd);
      int result = checkResponse();
      if (result != POSITIVE_INTERMEDIATE_REPLY) {
	controlOutputStream.println("ABOR\r");
	checkResponse();
      } else {
	tempCmd = new String("RNTO " + newName + "\r");
	printText(tempCmd);
	controlOutputStream.println(tempCmd);
	checkResponse();
      }
    }
  }
  
  
  /** 
   * Retrieve listing of the remote directory.
   *
   * @return A String array containing the set of 
   *         FileNameInfo objects. Each object contains
   *         the name of a file or directory and a type
   *         that specifies if it is a file or directory.
   
   public FileNameInfo[] list() {
   
   FileNameInfo tmpFileInfo;
   Vector tempVector = new Vector();
   String tempCmd;
   ServerSocket serverSocket = null;
   int result;
   
   if(State != CONNECTED) {
   printText("ERR Not connected, unable to list directories");
   return(new FileNameInfo[0]);
   }
   
   try {
   // Get a socket of the operating systems choosing
   serverSocket = new ServerSocket(0);
   } catch (IOException e) {
   printText("ERR Could not get port for listening: " +
   serverSocket.getLocalPort() + ", " + e);
   return(new FileNameInfo[0]);
   }
   
   setPort(serverSocket);
   
   // Now send the "list" command
   tempCmd = new String("NLST -al\r");
   printText(tempCmd);
   controlOutputStream.println(tempCmd);
   result = checkResponse();
   
   if(result == POSITIVE_PRELIMINARY_REPLY){
   // connect to data port
   Socket clientSocket = null;
   try {
   clientSocket = serverSocket.accept();
   } catch (IOException e) {
   printText("ERR Accept failed: " + 
   serverSocket.getLocalPort() + ", " + e);
   return(new FileNameInfo[0]);
   }
   
   try { 
   InputStream is = clientSocket.getInputStream();
   DataInputStream dis = new DataInputStream(is);
   String inLine;
   StringTokenizer sTok;
   boolean done = false;
   
   while( ! done ) {
   inLine = dis.readLine();
   if(inLine == null) { 
   done = true;
   } else {
   
   if( inLine.startsWith("total") ) {
   // Skip over the first line of output "total ..."
   } else {
   tmpFileInfo = new FileNameInfo(); 
   tmpFileInfo.parseUnixListing(inLine);
   tempVector.addElement(tmpFileInfo);
   }
   }
   }
   
   checkResponse();
   dis.close();
   clientSocket.close();
   } catch (IOException e) {
   printText("ERR " + e.toString());
   return(new FileNameInfo[0]);
   }
   } else {
   try { 
   serverSocket.close();
   } catch (IOException e) {
   printText("ERR " + e.toString());
   return(new FileNameInfo[0]);
   }
   printText("ERR Not able to get listing!");
   return(new FileNameInfo[0]);
   }
   
   // Create the array from the Vector
   FileNameInfo rVal[] = new FileNameInfo[tempVector.size()];
   tempVector.copyInto((Object[])rVal);
   
   return(rVal);
   }
   */
  
  /** 
   * Upload a file in binary mode using either stor or stou.  
   *
   * @param allowOverrite  True if the outgoing file should be allowed to 
   *                       overrite an existing remote file by the same name.
   * @param fileName       File name of the file to send to remote machine.
   *
   * @param proc           CpeProcess file.
   * @return Returns true on success.
   */
  public boolean sendFile(char type,
			  boolean allowOverrite, String fileName) {
	
    ServerSocket serverSocket = null;
    Socket clientSocket = null;
    OutputStream outdataport;
    String command;
    int result;
    int amount;
    
    
    if(State != CONNECTED) {
      printText("ERR Not connected, unable to send file!");
      return(false);
    }
    
    setType("BINARY");
    
    command = "PASV\r";
    printText(command);
    controlOutputStream.println(command);
    result = checkResponsePasv();

    try {
      clientSocket = new 
	Socket(Cpe.sharedInstance().hostName, m_port.intValue());  // ftp port
      outdataport = new DataOutputStream(clientSocket.getOutputStream());
    } catch (UnknownHostException e) {
      printText("ERR Don't know about host: " + 
		Cpe.sharedInstance().hostName );
      return(false);
    } catch (IOException e) {
      printText("ERR Couldn't get I/O for the connection to: " +
		Cpe.sharedInstance().hostName);
      return(false);
    }
    
    // ok, send command	
    if(allowOverrite) {
      command = "STOR " + fileName + "\r";
    } else {
      command = "STOU " + fileName + "\r";
    }
    controlOutputStream.println(command);
    printText(command);
    result = checkResponse();
    
    if(result == POSITIVE_PRELIMINARY_REPLY) {
      
      try {
	
	m_outdataport = outdataport;

	if (type == 'E')
	  exportDomainToFile(fileName);
	else
	  writeDomainToFile(fileName);

	m_outdataport = null;
	
	// clean up when done
	outdataport.close();
	clientSocket.close();

	result = checkResponse();
	
      } catch (IOException e) {
	printText("ERR " + e.toString() );
	return(false);
      }
      
      return(result == POSITIVE_COMPLETION_REPLY);
      
    } else {
      printText("ERR while trying to send file.");
      try {
	serverSocket.close();
      }
      catch (IOException e) {
	printText("ERR closing server socket.");
	return(false);
      }
      return(false);
    }
  }

  /** 
   * Save a file locally (not using Ftp) This should be moved to a diff class.
   *
   * @param type           Indicates a TF export or normal save...
   * @param fileName       File name of the file to send to remote machine.
   * @param os             Output stream
   *
   * @return Returns true on success.
   */
  public boolean sendFileLocal(char type, String fileName, OutputStream os) {
	
    int result;
    
    try {
	
      m_outdataport = os;

      if (type == 'E')
	exportDomainToFile(fileName);
      else
	writeDomainToFile(fileName);

      m_outdataport = null;
	
    } catch (Exception e) {
      printText("ERR " + e.toString() );
      return(false);
    }
    return(true);
  }
  
  
  /**
   * Performs downloading commands that need a data port.  Tells server
   * which data port it will listen on, then sends command on control port,
   * then listens on data port for server response.
   *
   * @param fileName       Name of file to get from local host.
   *
   * @return Returns true on success.
   */
  public boolean getFile(String fileName){
    
    String command; 
    int result;
    Socket clientSocket = null;
    DataInputStream is = null;
    //PrintStream os = null;
    
    setType("BINARY");
    
    command = "PASV\r";
    printText(command);
    controlOutputStream.println(command);
    result = checkResponsePasv();
    
    try {
      clientSocket = new 
	Socket(Cpe.sharedInstance().hostName, m_port.intValue());  // ftp port
      is  = new DataInputStream(clientSocket.getInputStream());
      //os = new PrintStream(clientSocket.getOutputStream());
    } catch (UnknownHostException e) {
      printText("ERR Don't know about host: " + 
		Cpe.sharedInstance().hostName );
      return(false);
    } catch (IOException e) {
      printText("ERR Couldn't get I/O for the connection to: " +
		Cpe.sharedInstance().hostName);
      return(false);
    }
    
    command = "RETR " + fileName + "\r";
    printText(command);
    controlOutputStream.println(command);
    result = checkResponse();
    
    if(result == POSITIVE_PRELIMINARY_REPLY) {

      Cpe.sharedInstance().displayMessage("Reading specification from file " +
					  fileName + " ...");      
      try { 
	//line = is.readLine();
        parseAndLoad(is);
	
	result = checkResponse();
	is.close();
	clientSocket.close();
	
      } catch (IOException e) {
	printText("ERR " + e.toString());
	Cpe.sharedInstance().displayMessage("ERR " + e.toString());
	return(false);
      }
      Cpe.sharedInstance().displayMessage("Read completed!");
      return(result == POSITIVE_COMPLETION_REPLY);
      
    } else {
      printText("ERR while trying to get file.");
      Cpe.sharedInstance().displayMessage("Err while trying to get the file.");
      return(false);
    }
    
  } 

  /**
   * Performs a local (non-ftp) file get. This should be moved to a new
   * class, but since most of the processing was here, I left it.
   * @param fileName       Name of file to get from local host.
   * @param is             Input stream.
   *
   * @return Returns true on success.
   */
  public boolean getFileLocal(String fileName, InputStream is){
    
    int result;

    Cpe.sharedInstance().displayMessage("Reading specification from file " +
					fileName + " ...");      
    try { 
      parseAndLoad(new DataInputStream(is));
    } catch (Exception e) {
      printText("ERR " + e.toString());
      Cpe.sharedInstance().displayMessage("ERR " + e.toString());
      return(false);
    }
    Cpe.sharedInstance().displayMessage("Read completed!");
    return(true);      
  } 
    
  /** 
   * CheckResponse from the control port after an ftp command has 
   * been sent.  Last line of the response begins with a 3 digit 
   * number folowed by a space. A dash would indicate a continuation.
   *
   * @return      returns the first character of the last line as an
   *              integer (ftpd return code)
   */
  public int checkResponse(){
    
    String str;
    
    try {
      do {
	str = controlInputStream.readLine();
	printText(str);
      } while(!(Character.isDigit(str.charAt(0)) &&
		Character.isDigit(str.charAt(1)) &&
		Character.isDigit(str.charAt(2)) && 
		str.charAt(3) == ' '));
    } catch (IOException e) {
      printText("ERR Error getting response from controlport!");
      return(0);
    }
    return(Integer.parseInt(str.substring(0,1)));
    
  } 
  
  
  /**
   *  Given the serverSocket, this method calculates the string to
   *  be sent to the ftp server as the first (and only) parameter
   *  to the ftp "port"  command. This "port" command is used to 
   *  notify the remote server of what data port we are waiting for
   *  a connection on.
   *
   *   @param  sock     Socket to get info from.
   *
   *   @return          string to use as the port command's
   *                    parameter or null (empty) string if error.
   */
  public String calcPortCmdParameter(ServerSocket sock) {
    String      rVal;
    int         localPort;
    InetAddress localAddr;
    InetAddress localIp;
    InetAddress [] adx;
    
    localPort = sock.getLocalPort();
    
    try {
      localIp = InetAddress.getLocalHost();
    } catch(UnknownHostException e) {
      return("");    // can't get local host address
    }
    
    // get ip address in high byte order
    byte[] addrbytes = localIp.getAddress();
    
    // tell server what port we are listening on
    short addrshorts[] = new short[4];
    
    // problem:  bytes greater than 127 are printed as negative numbers
    for(int i = 0; i <= 3; i++){
      addrshorts[i] = addrbytes[i];
      if(addrshorts[i] < 0) {
	addrshorts[i] += 256;
      }
    }
    
    rVal = new String(addrshorts[0] + "," + addrshorts[1] + "," + 
		      addrshorts[2] + "," + addrshorts[3] + "," + 
		      ((localPort & 0xff00) >> 8) + "," + 
		      (localPort & 0x00ff));
    
    // This is due to a problem while running on my home machine.
    // It keeps telling remote end to connect to 127.0.0.1 (localhost).
    /*
      rVal = new String("206,214,150,69," + 
      ((localPort & 0xff00) >> 8) + "," + 
      (localPort & 0x00ff));
      */
    
    return(rVal);
  }
  
  
  /** 
   * CheckResponse from the control port after an ftp command has 
   * been sent.  Last line of the response begins with a 3 digit 
   * number folowed by a space. A dash would indicate a continuation.
   *
   * @return      returns the first character of the last line as an
   *              integer (ftpd return code)
   */
  public int checkResponsePasv(){
    
    String str;
    Integer val1 = null;
    Integer val2 = null;
    int calcport = 0;
    
    try {
      do {
	str = controlInputStream.readLine();
	StringTokenizer st = new StringTokenizer(str, ",()");
	int i = 0;
	for  (Enumeration  e  = st; e.hasMoreElements()  ; i++)  {	
	  
	  String temp = (String) e.nextElement();
	  if (i == 5) 
	    {
	      val1 = new Integer(temp);
	    }
	  else if (i == 6) 
	    {
	      val2 = new Integer(temp);
	      //ex: 127,0,0,1,4,104 
	      //    -> 4 x 256 +104 = 1128
	      calcport = (val1.intValue() * 256) + val2.intValue();
	    }
	}
	
	m_port = new Integer(calcport);
	printText("Pasv port: " + m_port);
	printText(str);
	
      } while(!(Character.isDigit(str.charAt(0)) &&
		Character.isDigit(str.charAt(1)) &&
		Character.isDigit(str.charAt(2)) && 
		str.charAt(3) == ' '));
    } catch (IOException e) {
      printText("ERR Error getting response from controlport!");
      return(0);
    }
    return(Integer.parseInt(str.substring(0,1)));
    
  } 
  
  private void parseAndLoad (DataInputStream is) {

    Cpe.sharedInstance().cpdParse.spec.reset();
    Cpe.sharedInstance().cpdParse.ReInit(is);
    try {
      Cpe.sharedInstance().cpdParse.CompilationUnit();
    } catch (ParseException e) {
      Cpe.sharedInstance().displayMessage("Error! " + e.getMessage());
    } catch (TokenMgrError e1) {
      Cpe.sharedInstance().displayMessage("Error! " + e1.getMessage());
    } catch (Exception e2) {
      Cpe.sharedInstance().displayMessage("Error! " + e2.getMessage());
    }

    Cpe.sharedInstance().cpdParse.importer();
  }

  private void exportDomainToFile(String fileName) {

    Cpe.sharedInstance().displayMessage("Translating to TF");
    tfExport = true;
    tfOutputStream = new ByteArrayOutputStream();

    writeDomainToFile(fileName);
    Cpe.sharedInstance().tfExport = true;

    Cpe.sharedInstance().cpdParse.spec.reset();
    Cpe.sharedInstance().cpdParse.ReInit
      (new ByteArrayInputStream(tfOutputStream.toByteArray()));
    try {
      Cpe.sharedInstance().cpdParse.CompilationUnit();
    } catch (ParseException e) {
      Cpe.sharedInstance().displayMessage("Error! " + e.getMessage());
    } catch (TokenMgrError e1) {
      Cpe.sharedInstance().displayMessage("Error! " + e1.getMessage());
    }

    Cpe.sharedInstance().cpdParse.writeOut
      (new java.io.PrintStream(m_outdataport));
    Cpe.sharedInstance().tfExport = false;
    tfExport = false;
  }

  private void writeDomainToFile(String fileName) {
   
    Cpe.sharedInstance().displayMessage("Writing specification to file "+ 
					fileName + " ...");
    Cpe.sharedInstance().setWait();
    cpo_plan = new Vector();
    cpo_process = new Vector();
    cpo_activity_spec = new Vector();
    cpo_action = new Vector();
    cpo_start = new Vector();
    cpo_finish = new Vector();
    cpo_begin = new Vector();
    cpo_end = new Vector();
    cpo_timepoint = new Vector();
    cpo_ordering_c = new Vector();
    cpo_input_c = new Vector();
    cpo_output_c = new Vector();
    cpo_resource_c = new Vector();
    cpo_include_c = new Vector();
    cpo_annot_c = new Vector();
    cpo_unit = new Vector();
    cpo_var = new Vector();

    writeRemoteFile("//Common Domain Editor, version 1.0, DAI, 1998");
    writeRemoteFile("//Date: " + new Date());
    writeRemoteFile("//File: " + fileName);
    
    //writing system properties
    writeRemoteFile("//java.version: " +
		    System.getProperty("java.version"));
    writeRemoteFile("//java.vendor: " +
		    System.getProperty("java.vendor"));
    writeRemoteFile("//java.class.version: " +
		    System.getProperty("java.class.version"));
    writeRemoteFile("//os.name: " +
		    System.getProperty("os.name"));
    writeRemoteFile("//os.arch: " +
		    System.getProperty("os.arch"));
    writeRemoteFile("//os.version: " +
		    System.getProperty("os.version"));
    writeRemoteFile("//------------");

    writeDomainEntries();
    writeSortInfo();

    writeRemoteFile("//------------");
    writeRemoteFile("//End");

    cpo_plan = null;
    cpo_process = null;
    cpo_activity_spec = null;
    cpo_action = null;
    cpo_start = null;
    cpo_finish = null;
    cpo_begin = null;
    cpo_end = null;
    cpo_timepoint = null;
    cpo_ordering_c = null;
    cpo_input_c = null;
    cpo_output_c = null;
    cpo_resource_c = null;
    cpo_include_c = null;
    cpo_annot_c = null;
    cpo_unit = null;
    cpo_var = null;

    Cpe.sharedInstance().displayMessage("Save completed!");
    Cpe.sharedInstance().setNormal();
  }

  private void writeSortInfo() {
    int i=0;
    String sortList = "";

    //cpo_plan
    i=0;sortList="";
    for  (Enumeration  e  = cpo_plan.elements(); e.hasMoreElements();i++)  {
      String temp = (String) e.nextElement();
      if (i==0) sortList = temp;
      else sortList = sortList + "," + temp;
    }
    writeRemoteFile("SORT cpo-plan={"+sortList+"}");

    //cpo_process
    i=0;sortList="";
    for  (Enumeration  e  = cpo_process.elements(); e.hasMoreElements();i++) {
      String temp = (String) e.nextElement();
      if (i==0) sortList = temp;
      else sortList = sortList + "," + temp;
    }
    writeRemoteFile("SORT cpo-process={"+sortList+"}");

    //cpo_activity_spec
    i=0;sortList="";
    for  (Enumeration  e  = cpo_activity_spec.elements(); 
	  e.hasMoreElements();i++) {
      String temp = (String) e.nextElement();
      if (i==0) sortList = temp;
      else sortList = sortList + "," + temp;
    }
    writeRemoteFile("SORT cpo-activity-specification={"+sortList+"}");

    //cpo_action
    i=0;sortList="";
    for  (Enumeration  e  = cpo_action.elements(); e.hasMoreElements();i++) {
      String temp = (String) e.nextElement();
      if (i==0) sortList = temp;
      else sortList = sortList + "," + temp;
    }
    writeRemoteFile("SORT cpo-action={"+sortList+"}");

    //cpo_start
    i=0;sortList="";
    for  (Enumeration  e  = cpo_start.elements(); e.hasMoreElements();i++) {
      String temp = (String) e.nextElement();
      if (i==0) sortList = temp;
      else sortList = sortList + "," + temp;
    }
    writeRemoteFile("SORT cpo-start={"+sortList+"}");

    //cpo_finish
    i=0;sortList="";
    for  (Enumeration  e  = cpo_finish.elements(); e.hasMoreElements();i++) {
      String temp = (String) e.nextElement();
      if (i==0) sortList = temp;
      else sortList = sortList + "," + temp;
    }
    writeRemoteFile("SORT cpo-finish={"+sortList+"}");

    //cpo_begin
    i=0;sortList="";
    for  (Enumeration  e  = cpo_begin.elements(); e.hasMoreElements();i++) {
      String temp = (String) e.nextElement();
      if (i==0) sortList = temp;
      else sortList = sortList + "," + temp;
    }
    writeRemoteFile("SORT cpo-begin={"+sortList+"}");

    //cpo_end
    i=0;sortList="";
    for  (Enumeration  e  = cpo_end.elements(); e.hasMoreElements();i++) {
      String temp = (String) e.nextElement();
      if (i==0) sortList = temp;
      else sortList = sortList + "," + temp;
    }
    writeRemoteFile("SORT cpo-end={"+sortList+"}");

    //cpo_timepoint
    i=0;sortList="";
    for  (Enumeration e  = cpo_timepoint.elements(); e.hasMoreElements();i++) {
      String temp = (String) e.nextElement();
      if (i==0) sortList = temp;
      else sortList = sortList + "," + temp;
    }
    writeRemoteFile("SORT cpo-timepoint={"+sortList+"}");

    //cpo_ordering_c
    i=0;sortList="";
    for  (Enumeration e = cpo_ordering_c.elements(); e.hasMoreElements();i++) {
      String temp = (String) e.nextElement();
      if (i==0) sortList = temp;
      else sortList = sortList + "," + temp;
    }
    writeRemoteFile("SORT cpo-ordering-constraint={"+sortList+"}");

    //cpo_input_c
    i=0;sortList="";
    for  (Enumeration e = cpo_input_c.elements(); e.hasMoreElements();i++) {
      String temp = (String) e.nextElement();
      if (i==0) sortList = temp;
      else sortList = sortList + "," + temp;
    }
    writeRemoteFile("SORT cpo-input-constraint={"+sortList+"}");

    //cpo_output_c
    i=0;sortList="";
    for  (Enumeration e = cpo_output_c.elements(); e.hasMoreElements();i++) {
      String temp = (String) e.nextElement();
      if (i==0) sortList = temp;
      else sortList = sortList + "," + temp;
    }
    writeRemoteFile("SORT cpo-output-constraint={"+sortList+"}");

    //cpo_resource_c
    i=0;sortList="";
    for  (Enumeration e = cpo_resource_c.elements(); e.hasMoreElements();i++) {
      String temp = (String) e.nextElement();
      if (i==0) sortList = temp;
      else sortList = sortList + "," + temp;
    }
    writeRemoteFile("SORT cpo-resource-constraint={"+sortList+"}");

    //cpo_unit
    i=0;sortList="";
    for  (Enumeration e = cpo_unit.elements(); e.hasMoreElements();i++) {
      String temp = (String) e.nextElement();
      if (i==0) sortList = temp;
      else sortList = sortList + "," + temp;
    }
    writeRemoteFile("SORT cpo-resource-unit={"+sortList+"}");

    //cpo_var
    i=0;sortList="";
    for  (Enumeration e = cpo_var.elements(); e.hasMoreElements();i++) {
      String temp = (String) e.nextElement();
      if (i==0) sortList = temp;
      else sortList = sortList + "," + temp;
    }
    writeRemoteFile("SORT cpo-entity-variable={"+sortList+"}");

    //cpo_include_c
    i=0;sortList="";
    for  (Enumeration e  = cpo_include_c.elements(); e.hasMoreElements();i++) {
      String temp = (String) e.nextElement();
      if (i==0) sortList = temp;
      else sortList = sortList + "," + temp;
    }
    writeRemoteFile("SORT cpo-include-constraint={"+sortList+"}");

    //cpo_annot_c
    i=0;sortList="";
    for  (Enumeration e  = cpo_annot_c.elements(); e.hasMoreElements();i++) {
      String temp = (String) e.nextElement();
      if (i==0) sortList = temp;
      else sortList = sortList + "," + temp;
    }
    writeRemoteFile("SORT cpo-annotation-constraint={"+sortList+"}");
  }

  private void writeDomainEntries() {

    DefaultMutableTreeNode node;
    DefaultMutableTreeNode root = (DefaultMutableTreeNode) 
      tree.getModel().getRoot();
    CpeTreeItem item;

    int i = 0; //only go thru the toplevel items (4)
    for  (Enumeration e = root.breadthFirstEnumeration(); 
	  e.hasMoreElements() && i<4;i++)  {	
      node = (DefaultMutableTreeNode) e.nextElement();
      item = (CpeTreeItem) node.getUserObject();
      item.writeOut(this,node);
    }
  }

  private void writeSingleProcess(CpeProcessPanel frame) {
  }

  public void writeRemoteFile(String string) {
    byte b[] = new byte[1024];  // 1K blocks I guess

    string = string + '\n';
    int amount = string.length();
    b = string.getBytes();
    try {
      if (tfExport)
	tfOutputStream.write(b, 0, amount);
      else
	m_outdataport.write(b, 0, amount);
    } catch (IOException e) {
      printText("ERR with writeRemoteFile");
    }

  }

  public void writeFunction(String functerm, String element, String value) {
    writeRemoteFile(functerm +"("+element+")="+value);
  }
  public void writeFunctionStr(String functerm,String element,String value) {
    writeRemoteFile(functerm +"("+element+")=\""+value+"\"");
  }
  public void write2Relation(String pred, String term1, String term2) {
    writeRemoteFile(pred +"("+term1+","+term2+")");
  }
  public void write3Relation(String pred, String term1, String term2,
			     int term3) {
    writeRemoteFile(pred +"("+term1+","+term2+","+
		    (new Integer(term3)).toString() + ")");
  }
}

