/*
* @(#)CpeRation.java	1.0 98/1/5
* 
*/

import java.util.*;
import java.awt.*;
import java.awt.event.*;
import CpeNode;
import CpeQNode;
import CpeONode;
import CpeCNode;
import CpeEdge;
import CpeEqEdge;
import CpeQOEdge;
import CpePosEdge;
import CpeNegEdge;
import CpeRationPanel;
import com.sun.java.swing.*;
import CpeRatNodePropertyDialog;
import CpeRatProcPropertyDialog;

/**
* CpeRation encapsulates the main rationale data structures and is reponsible
* for the display and management of the process. 
*
* @version 1.0 98/1/5
* @author Steve Polyak 
*/
class CpeRation {
	
  //Elements kept as a vector of nodes, edges, and text items
  Vector m_nodes;
  Vector m_edges;
  Vector m_text;
	
  //For now, additional elements as vectors of strings
  Vector preconditions;
  Vector effects;
  Vector requirements;
  Vector preferences;
  Vector variables;
  Vector resources;
	
  //Process has a key, label, and an expands ("" for top level)
  String key = "";
  String label = "";
  String expands = "";
	
  //Some Processes are top-level (and their label is the same on the top tab)
  boolean topLevelFlag = false;
  boolean startFinishFlag = true;
	
  //Store pointers to the process panel and dialogs
  CpeRationPanel m_panel = null;
  CpeRatNodePropertyDialog m_sheet = null;
  CpeRatProcPropertyDialog processSheet = null;
  JPopupMenu popup = null;
	
  //Constants
  final static String FINISH_STRING = "Finish";
  final static String START_STRING = "Start";
  final static String BEGIN_STRING = "Begin";
  final static String END_STRING = "End";
	
  //Graphics elements
  final Color m_edgeColor = Color.black;
  final Color m_linkColor = Color.red;
  Dimension m_dimension;
  Rectangle m_selectingRect;
	
  //Flags, Counters, Temp Variables
  long m_nodeCounter = 1;
  boolean hasRationale = false;
  boolean m_linkDrawing = false;
  int m_linkType = 0; //0=QO,1=Pos,2=Neg,3=OQ
  boolean m_linkBegin = false;
  CpeNode m_hitNode = null;
  boolean m_selecting = false;
  int m_linkToX = 0;
  int m_linkToY = 0;
  Point popupPoint = new Point();
  String rationale = "";

  //Constructor saves a reference back to the panel.
  CpeRation(CpeRationPanel panel, boolean top) {
    m_panel = panel;
    m_nodes = new Vector();
    m_edges = new Vector();
    m_text = new Vector();
    preconditions = new Vector();
    effects = new Vector();
    requirements = new Vector();
    preferences = new Vector();
    variables = new Vector();
    resources = new Vector();
		
    topLevelFlag = top;
  }
	
  public void clear() {
    m_edges.removeAllElements();
    m_nodes.removeAllElements();  
    m_text.removeAllElements();  
    preconditions.removeAllElements();
    effects.removeAllElements();
    requirements.removeAllElements();
    preferences.removeAllElements();
    variables.removeAllElements();
    resources.removeAllElements();
  }
	
  public void newProcess() {
		
    m_dimension = m_panel.size();
    clear();
  }
	
  void addNode(CpeNode node) {
    m_nodes.addElement(node);
  }
	
  void addNewNode(CpeNode node) {
    CpeNode newNode = new CpeNode(node.m_key, node.m_x, node.m_y);
    newNode.setStatus(node.m_status);
    newNode.m_lbl = node.m_lbl;
    //Note: I am omitting expansion stuff, too complex for now :-)
    addNode(newNode);
  }
	
  void addNewNode(CpeSpecialNode node) {
    CpeSpecialNode newNode = new CpeSpecialNode(node.m_key, 
						node.m_x, node.m_y);
    newNode.setStatus(node.m_status);
    newNode.m_lbl = node.m_lbl;
    //Note: I am omitting expansion stuff, too complex for now :-)
    addNode(newNode);
  }
	
  void removeSelected() {
		
    Cpe cpe = Cpe.sharedInstance();
    cpe.clearCopySpace();
		
    Vector delNodes = new Vector();
    CpeNode tempNode;
    for  (Enumeration e =  m_nodes.elements() ; 
	  e.hasMoreElements()  ;)  {	
      tempNode = (CpeNode) e.nextElement();
      if (tempNode.m_selected) {
	delNodes.addElement(tempNode);
	cpe.cpNodes.addElement(tempNode);
      }
    }
    for  (Enumeration  e2  =  delNodes.elements() ;
	  e2.hasMoreElements()  ;)  {	
      tempNode = (CpeNode) e2.nextElement();
      removeNode(tempNode);
    }
    delNodes = null;
		
    //remove edges
    Vector delEdges = new Vector();
    CpeEdge tempEdge;
    for  (Enumeration e3 =  m_edges.elements() ; 
	  e3.hasMoreElements()  ;)  {	
      tempEdge = (CpeEdge) e3.nextElement();
      if (tempEdge.m_selected) {
	delEdges.addElement(tempEdge);
	cpe.cpEdges.addElement(tempEdge);
      }
    }
    for  (Enumeration  e4  =  delEdges.elements() ;
	  e4.hasMoreElements()  ;)  {	
      tempEdge = (CpeEdge) e4.nextElement();
      m_edges.removeElement(tempEdge);
    }
    delEdges = null;
		
		
    //remove text
    Vector delText = new Vector();
    CpeText tempText;
    for  (Enumeration e5 =  m_text.elements() ; 
	  e5.hasMoreElements()  ;)  {	
      tempText = (CpeText) e5.nextElement();
      if (tempText.m_selected) {
	delText.addElement(tempText);
	cpe.cpText.addElement(tempText);
      }
    }
    for  (Enumeration  e6  =  delText.elements() ;
	  e6.hasMoreElements()  ;)  {	
      tempText = (CpeText) e6.nextElement();
      m_text.removeElement(tempText);
    }
    delText = null;
		
  }
	
  public void copySelected() {
		
    Cpe cpe = Cpe.sharedInstance();
    cpe.clearCopySpace();
		
    //copy nodes
    CpeNode tempNode;
    for  (Enumeration e =  m_nodes.elements() ; 
	  e.hasMoreElements()  ;)  {	
      tempNode = (CpeNode) e.nextElement();
      if (tempNode.m_selected) {
	cpe.cpNodes.addElement(tempNode);
      }
    }
		
    //copy edges
    Vector delEdges = new Vector();
    CpeEdge tempEdge;
    for  (Enumeration e3 =  m_edges.elements() ; 
	  e3.hasMoreElements()  ;)  {	
      tempEdge = (CpeEdge) e3.nextElement();
      if (tempEdge.m_selected) {
	cpe.cpEdges.addElement(tempEdge);
      }
    }
		
    //copy text
    Vector delText = new Vector();
    CpeText tempText;
    for  (Enumeration e5 =  m_text.elements() ; 
	  e5.hasMoreElements()  ;)  {	
      tempText = (CpeText) e5.nextElement();
      if (tempText.m_selected) {
	cpe.cpText.addElement(tempText);
      }
    }
  }
	
  public void pasteSelected() {
		
    Cpe cpe = Cpe.sharedInstance();
		
    //paste nodes
    CpeNode tempNode;
    for  (Enumeration e =  cpe.cpNodes.elements() ; 
	  e.hasMoreElements()  ;)  {	
      tempNode = (CpeNode) e.nextElement();
      if (locateNode(tempNode.m_key) == null) {
	//yuck! should be fixed later
	if ("CpeSpecialNode".equals(tempNode.getClass().getName()))
	  addNewNode((CpeSpecialNode)tempNode);
	else
	  addNewNode(tempNode);
      }
    }
		
    //paste edges (if possible)
    CpeEdge tempEdge;
    for  (Enumeration e3 =  cpe.cpEdges.elements() ; 
	  e3.hasMoreElements()  ;)  {	
      tempEdge = (CpeEdge) e3.nextElement();
      if ((locateNode(tempEdge.m_from.m_key) != null) &&
	  (locateNode(tempEdge.m_to.m_key) != null) &&
	  (locateEdge(tempEdge) == null)) {
	//yuck! should be fixed later
	if ("CpeEqEdge".equals(tempEdge.getClass().getName()))
	  addEdge((CpeEqEdge)tempEdge);
	else
	  addEdge(tempEdge);
      }
    }
		
    //paste text
    CpeText tempText;
    for  (Enumeration e5 =  cpe.cpText.elements() ; 
	  e5.hasMoreElements()  ;)  {	
      tempText = (CpeText) e5.nextElement();
      m_text.addElement(new CpeText(tempText.m_key, tempText.m_x,
				    tempText.m_y));
    }
  }
	
  void removeNode(CpeNode node) {
    CpeEdge tempEdge;
		
    //cannot delete bounding nodes on top level
    if ((node.m_key.equals("")) || 
	(node.m_key.equals(START_STRING)) ||
	(node.m_key.equals(BEGIN_STRING)) ||
	(node.m_key.equals(END_STRING)) ||
	(node.m_key.equals(FINISH_STRING))) {
      return;
    }
		
    //no need to search for node, just search 
    // for any edges.
    Vector delEdges = new Vector();
    for  (Enumeration  e1  =  m_edges.elements() ;
	  e1.hasMoreElements()  ;)  {	
      tempEdge = (CpeEdge) e1.nextElement();
      if ((tempEdge.m_from.m_key.equals(node.m_key)) || 
	  (tempEdge.m_to.m_key.equals(node.m_key))) {
	delEdges.addElement(tempEdge);
      }
    }
    for  (Enumeration  e2  =  delEdges.elements() ; 
	  e2.hasMoreElements()  ;)  {	
      tempEdge = (CpeEdge) e2.nextElement();
      m_edges.removeElement(tempEdge);
    }
    m_nodes.removeElement(node);
    delEdges = null;
  }
	
  void renameNode(String from, String to) 
  {
    CpeNode temp;
		
    //can't rename bounding nodes
    if ((from.equals("")) || 
	(to.equals("")) ||
	(from.equals(START_STRING)) ||
	(to.equals(START_STRING)) ||
	(from.equals(FINISH_STRING)) ||
	(to.equals(FINISH_STRING)) ||
	(from.equals(BEGIN_STRING)) ||
	(to.equals(BEGIN_STRING)) ||
	(from.equals(END_STRING)) ||
	(to.equals(END_STRING)) ||
	(locateNode(to) != null)) {
      return;
    }
		
    //For now, a simple linear search
    for  (Enumeration  e  =  m_nodes.elements() ;
	  e.hasMoreElements()  ;)  {	
      temp = (CpeNode) e.nextElement();
      if (temp.m_key.equals(from)) {
	temp.m_lbl = to;
	break; 
      }
    }
  }
	
  CpeNode locateNode(String key) {
    CpeNode found = null;
    CpeNode temp;
		
    //For now, a simple linear search
    for  (Enumeration  e  =  m_nodes.elements() ;
	  e.hasMoreElements()  ;)  {	
      temp = (CpeNode) e.nextElement();
      if (temp.m_key.equals(key)) {
	found = temp;
	break; 
      }
    }	
    return found;
  }
	
  void addEdge(CpeEdge edge) {
    addEdge(edge.m_from.m_key, edge.fromPosition, edge.m_to.m_key,
	    edge.toPosition);
  }
	
  void addEdge(CpeEqEdge edge) {
    addEqEdge(edge.m_from.m_key, edge.fromPosition, edge.m_to.m_key,
	      edge.toPosition);
  }
	
  void addEdge(String from, boolean fromBegin, String to, boolean toBegin) {
    CpeNode nodeFrom, nodeTo;
    nodeFrom = locateNode(from);
    nodeTo = locateNode(to);
		
    m_edges.addElement(new CpeEdge(nodeFrom, fromBegin, nodeTo, toBegin));
  }
	
  void addEqEdge(String from, boolean fromBegin, String to, boolean toBegin) {
    CpeNode nodeFrom, nodeTo;
    nodeFrom = locateNode(from);
    nodeTo = locateNode(to);
		
    m_edges.addElement(new CpeEqEdge(nodeFrom, fromBegin, nodeTo, toBegin));
  }

  void addQOEdge(String from, boolean fromBegin, String to, boolean toBegin) {
    CpeNode nodeFrom, nodeTo;
    nodeFrom = locateNode(from);
    nodeTo = locateNode(to);
		
    m_edges.addElement(new CpeQOEdge(nodeFrom, fromBegin, nodeTo, toBegin));
  }

  void addPosEdge(String from, boolean fromBegin, String to, boolean toBegin) {
    CpeNode nodeFrom, nodeTo;
    nodeFrom = locateNode(from);
    nodeTo = locateNode(to);
		
    m_edges.addElement(new CpePosEdge(nodeFrom, fromBegin, nodeTo, toBegin));
  }

  void addNegEdge(String from, boolean fromBegin, String to, boolean toBegin) {
    CpeNode nodeFrom, nodeTo;
    nodeFrom = locateNode(from);
    nodeTo = locateNode(to);
		
    m_edges.addElement(new CpeNegEdge(nodeFrom, fromBegin, nodeTo, toBegin));
  }
	
  public void removeEdge(String from, boolean fromBegin,
			 String to, boolean toBegin) {
    CpeEdge temp;
    //For now, a simple linear search
    for  (Enumeration  e  =  m_edges.elements() ;
	  e.hasMoreElements()  ;)  {	
      temp = (CpeEdge) e.nextElement();
      if ((temp.m_from.m_key.equals(from)) &&
	  (temp.fromPosition == fromBegin) &&
	  (temp.toPosition == toBegin) &&
	  (temp.m_to.m_key.equals(to))) {
	m_edges.removeElement(temp);
	break; 
      }
    }
  }
	
  public CpeEdge locateEdge(CpeEdge edge) {
    return locateEdge(edge.m_from.m_key, edge.fromPosition, 
		      edge.m_to.m_key, edge.toPosition);
  }
	
  public CpeEdge locateEdge(String from, boolean fromBegin, 
			    String to, boolean toBegin ) {
    CpeEdge found = null;
    CpeEdge temp;
		
    //For now, a simple linear search
    for  (Enumeration  e  =  m_edges.elements() ;
	  e.hasMoreElements()  ;)  {	
      temp = (CpeEdge) e.nextElement();
      if ((temp.m_from.m_key.equals(from)) &&
	  (temp.fromPosition == fromBegin) &&
	  (temp.m_to.m_key.equals(to)) &&
	  (temp.toPosition == toBegin)) {
	found = temp;
	break; 
      }
    }	
    return found;
  }
	
  void update(Graphics g ) {
		
    Dimension d = m_panel.size();    
		
    g.setColor(m_panel.getBackground());
    g.fillRect(0, 0, d.width, d.height);
    FontMetrics fm = g.getFontMetrics();
		
    // Draw each edge
    for  (Enumeration  e  =  m_edges.elements() ;
	  e.hasMoreElements()  ;)  {	
      ((CpeEdge) e.nextElement()).draw(g, fm);
    }	
		
    // Draw each node.
    for  (Enumeration  e  =  m_nodes.elements() ;
	  e.hasMoreElements()  ;)  {	
      ((CpeNode) e.nextElement()).draw(g, fm);
    }
		
    // Draw each CpeEdge arrow
    for  (Enumeration  e  =  m_edges.elements() ;
	  e.hasMoreElements()  ;)  {	
      ((CpeEdge) e.nextElement()).drawArrow(g, fm);
    }	
		
    // Draw each annotation.
    for  (Enumeration  e  =  m_text.elements() ;
	  e.hasMoreElements()  ;)  {	
      ((CpeText) e.nextElement()).draw(g, fm);
    }
		
    //Link drawing and selection goes on top.
    if (m_linkDrawing) {
      g.setColor(m_linkColor);
      if (m_linkBegin) {
	g.drawLine(m_panel.m_hitNode.m_rect.x + 10,
		   m_panel.m_hitNode.m_y,
		   m_linkToX, m_linkToY);
      }
      else {
	g.drawLine(m_panel.m_hitNode.m_rect.x +
		   m_panel.m_hitNode.m_rect.width - 10,
		   m_panel.m_hitNode.m_y,
		   m_linkToX, m_linkToY);
      }
    }
		
    if (m_selecting) {
      Rectangle box = getDrawableRect(m_selectingRect, d);
      g.setColor(m_linkColor);
      g.drawRect(box.x, box.y, box.width - 1, box.height - 1);
    }
  }
	
  public boolean mouseDown(MouseEvent evt) {
    CpeNode temp;
    m_hitNode = null;
    boolean hitText = false;
    int x = evt.getX();
    int y = evt.getY();
		
    if(m_linkDrawing) {
      m_linkDrawing = false;
      linkDrawDown(x,y);
    }
		
    //look for a hit CpeText (in reverse for proper Z order)
    for(int intSize = m_text.size(); intSize > 0; intSize--) {
      temp = (CpeText) m_text.elementAt(intSize - 1);
      if (temp.m_rect.inside(x,y)) {
	m_hitNode = temp;
	hitText = true;
	break;	
      }
    }
		
    if (hitText == false) {
      //look for a hit CpeNode (in reverse for proper Z order)
      for(int intSize = m_nodes.size(); intSize > 0; intSize--) {
	temp = (CpeNode) m_nodes.elementAt(intSize - 1);
	if (temp.m_rect.inside(x,y)) {
	  m_hitNode = temp;
	  break;	
	}
      }
    }
		
    if ((evt.isMetaDown()) && (hitText != true)) {
      createPopup();
      popup.show(m_panel,x,y);
      popupPoint.x = x; 
      popupPoint.y = y;
      return true;
    }
    else if (m_hitNode != null) {
      if (evt.isControlDown() == false)
	deselectAll();
      m_hitNode.m_selected = true;
    }
    else {
      //going into selection mode
      deselectAll();
      m_selecting = true;
      m_selectingRect = new Rectangle(x, y, 0, 0);
    }
    m_panel.repaint();
    return true;
  }
	
  public boolean mouseDrag(MouseEvent evt) {
    int x = evt.getX();
    int y = evt.getY();
		
    if (m_panel.inside(x,y)) {
      if (m_selecting) {
	m_selectingRect.resize(x - m_selectingRect.x,
			       y - m_selectingRect.y);
      }
      else {
	if ((m_hitNode != null) &&
	    (evt.isMetaDown() == false)) {
	  m_hitNode.m_x = x;
	  m_hitNode.m_y = y;
	}
      }
    }
    m_panel.repaint();
    return true;
  }
	
  public boolean mouseMove(MouseEvent evt) {
		
    if(m_selecting) {
      m_selecting = false;
      m_panel.repaint();
    }
    if(m_linkDrawing) {
      m_linkToX = evt.getX();
      m_linkToY = evt.getY();
      m_panel.repaint();
    }
    return true;
  }
	
  public boolean mouseUp(MouseEvent evt) {
		
    int x = evt.getX();
    int y = evt.getY();
		
    if (m_panel.inside(x,y) && (m_selecting)) {
      m_selecting = false;
      selectingUp(x,y);
    }
    m_hitNode = null;
    m_panel.repaint();
    return true;
  }
	
  void linkDrawDown(int x, int y) {
    CpeNode tempNode;
    CpeEdge tempEdge;
		
    //look for the CpeNode that was released on
    for(int intSize = m_nodes.size(); intSize > 0; intSize--) {
      tempNode = (CpeNode) m_nodes.elementAt(intSize - 1);
      if (tempNode.m_rect.inside(x,y)) {
	boolean toBegin = tempNode.isBegin(new Point(x,y));
	//found node, make sure CpeEdge doesn't exist
	if (locateEdge(m_panel.m_hitNode.m_key, m_linkBegin,
		       tempNode.m_key, toBegin) == null) {
	  if (m_linkType == 0)
	    addQOEdge(m_panel.m_hitNode.m_key, m_linkBegin,
		      tempNode.m_key, toBegin);
	  else if (m_linkType == 1)
	    addPosEdge(m_panel.m_hitNode.m_key, m_linkBegin,
		       tempNode.m_key, toBegin);
	  else if (m_linkType == 2)
	    addNegEdge(m_panel.m_hitNode.m_key, m_linkBegin,
		       tempNode.m_key, toBegin);

	  else
	    addEdge(m_panel.m_hitNode.m_key, m_linkBegin,
		    tempNode.m_key, toBegin);
	}
	break;	
      }
    }
  }
	
  void selectingUp(int x, int y) {
    CpeNode tempNode;
    CpeText tempText;
    CpeEdge tempEdge;
    m_dimension = m_panel.size();
    Rectangle box = getDrawableRect(m_selectingRect, m_dimension);
		
    //find nodes that intersect with the selection
    for  (Enumeration  e  =  m_nodes.elements() ;
	  e.hasMoreElements()  ;)  {	
      tempNode = (CpeNode) e.nextElement();
      if (tempNode.m_rect.intersects(box)){
	tempNode.m_selected = true;
      }
    }
		
    //find text that intersect with the selection
    for  (Enumeration  e2  =  m_text.elements() ;
	  e2.hasMoreElements()  ;)  {	
      tempText = (CpeText) e2.nextElement();
      if (tempText.m_rect.intersects(box)){
	tempText.m_selected = true;
      }
    }
		
    //find edges that intersect
    for  (Enumeration  e1  =  m_edges.elements() ;
	  e1.hasMoreElements()  ;)  {	
      tempEdge = (CpeEdge) e1.nextElement();
      //rough heuristic checks that selection could intersect line
      if(tempEdge.getRect().intersects(box)) {
	//need to check the points along the line 
	if (checkEdgeHit(tempEdge,box)) {
	  tempEdge.m_selected = true;
	}
      }
    }
  }
	
  public void addDummy()
  {
    String newString = new String("Dummy" );
    newString = newString.concat(Long.toString(m_nodeCounter));
    addNode(new CpeSpecialNode(newString, popupPoint.x, popupPoint.y));
    m_nodeCounter++;
  }

  public void addQuestion()
  {
    String newString = new String("Question" );
    newString = newString.concat(Long.toString(m_nodeCounter));
    addNode(new CpeQNode(newString, popupPoint.x, popupPoint.y));
    m_nodeCounter++;
  }

  public void addOption()
  {
    String newString = new String("Option" );
    newString = newString.concat(Long.toString(m_nodeCounter));
    addNode(new CpeONode(newString, popupPoint.x, popupPoint.y));
    m_nodeCounter++;
  }

  public void addCriteria()
  {
    String newString = new String("Criteria" );
    newString = newString.concat(Long.toString(m_nodeCounter));
    addNode(new CpeCNode(newString, popupPoint.x, popupPoint.y));
    m_nodeCounter++;
  }
	
  public void addActivity(){
    String newString = new String("act" );
    newString = newString.concat(Long.toString(m_nodeCounter++));
    String leftBoundString = null;
    String rightBoundString = null;
		
    if (startFinishFlag == true) {
      leftBoundString = START_STRING;
      rightBoundString = FINISH_STRING;
    } else {
      leftBoundString = BEGIN_STRING;
      rightBoundString = END_STRING;
    }
		
    if ((m_nodes.size() < 3) &&
	(locateNode(leftBoundString) != null) &&
	(locateNode(rightBoundString) != null)){
      removeEdge(leftBoundString, false, rightBoundString, true);
      Point pt = new Point();
      pt = getMiddlePoint(locateNode(leftBoundString), 
			  locateNode(rightBoundString));
      addNode(new CpeNode(newString, pt.x, pt.y));
      addEdge(leftBoundString, false, newString, true);
      addEdge(newString, false, rightBoundString, true);
    }
    else {
      addNode(new CpeNode(newString, popupPoint.x, popupPoint.y));
    }
  }
	
  Rectangle getDrawableRect(Rectangle originalRect, 
			    Dimension drawingArea) {
    int x = originalRect.x;
    int y = originalRect.y;
    int width = originalRect.width;
    int height = originalRect.height;
		
    //Make sure rectangle width and height are positive.
    if (width < 0) {
      width = 0 - width;
      x = x - width + 1;
      if (x < 0) {
	width += x;
	x = 0;
      }
    }
    if (height < 0) {
      height = 0 - height;
      y = y - height + 1;
      if (y < 0) {
	height += y;
	y = 0;
      }
    }
		
    //The rectangle shouldn't extend past the drawing area.
    if ((x + width) > drawingArea.width) {
      width = drawingArea.width - x;
    }
    if ((y + height) > drawingArea.height) {
      height = drawingArea.height - y;
    }
    return new Rectangle(x, y, width, height);
  }
	
  void deselectAll() {
    for  (Enumeration  e  =  m_nodes.elements() ;
	  e.hasMoreElements()  ;)  {	
      ((CpeNode) e.nextElement()).m_selected = false;
    }
    for  (Enumeration  e1  =  m_edges.elements() ;
	  e1.hasMoreElements()  ;)  {	
      ((CpeEdge) e1.nextElement()).m_selected = false;
    }
    for  (Enumeration  e1  =  m_text.elements() ;
	  e1.hasMoreElements()  ;)  {	
      ((CpeText) e1.nextElement()).m_selected = false;
    }
  }
	
  private Point getMiddlePoint(CpeNode node1, CpeNode node2) {
		
    //calc point halfway between the two nodes
    Point pt = new Point();
		
    if (node1.m_x >= node2.m_x)
      pt.x = node2.m_x + ((int)(node1.m_x - node2.m_x)/2);
    else
      pt.x = node1.m_x + ((int)(node2.m_x - node1.m_x)/2);
    if (node1.m_y >= node2.m_y)
      pt.y = node2.m_y + ((int)(node1.m_y - node2.m_y)/2);
    else
      pt.y = node1.m_y + ((int)(node2.m_y - node1.m_y)/2);
		
    return pt;
  }
	
  //updates the status bar description
  public void updateStatus(String message, Color color) { 
    //todo: add a status bar
  }
	
  private void createPopup() {
    int ibegin, iend;
    String[] labels = new String[] { 
      "Add Before Link", "Add Equals Link", "Add Expansion", 
	"-", "Show Expansion", "Edit Rationale", "-",
	"Delete Expansion", "Delete Node", "-", "Properties", 
	"Add Activity", "Add Dummy", "Add Text", "-", "Delete Selection", 
	"Edit Rationale", "-", "Properties"};
		
    String[] commands = new String[] {
      "addlink", "addeqlink", "addexp", "-", "showexp", "nration", 
	"-", "delexp", "delnode", "-", "nodeprop", "newact", 
	"newdummy", "addtext", "-", "delsel", "pration", "-", "procprop"};
				
    popup = new JPopupMenu();
    if (m_hitNode  != null) {
      //meta-clicked on a node
      ibegin = 0;
      iend = 10;
      m_panel.m_hitNode = m_hitNode;
    }
    else {
      //meta-clicked on the panel space
      ibegin = 11;
      iend = labels.length - 1;
    }
    for(int i=ibegin; i <= iend; i++) {
      if (labels[i].equals("-")) {
	popup.addSeparator();
      }
      else {
	JMenuItem mi = new JMenuItem(labels[i]);
	mi.setActionCommand(commands[i]);
	mi.addActionListener(m_panel);
	popup.add(mi);
      }
    }
    m_panel.add(popup);
  }
	
	
  private boolean checkEdgeHit(CpeEdge edge, Rectangle box) {
    int x1, y1, x2, y2;
    double m, tempY, tempX, ystart, yend, b;
		
    x1 = edge.fromX();
    y1 = edge.fromY();
    x2 = edge.toX();
    y2 = edge.toY();
		
    //first, check if either begin or end point are in selection area
    if (box.inside(x1,y1) || box.inside(x2,y2))
      return true;
		
    //calculating points on the line using y = mx+b
    //calc slope (m)
    if (x1-x2 == 0) {
      //no change in x, only check y points
      return checkEdgeHity(y1,y2,x1,box);
    }
		
    m = (double) (y1-y2)/(x1-x2);
    if (m == 0.0) {
      //no change in y, only check x points
      return checkEdgeHitx(x1,x2,y1,box);
    }
    b = (double) (-1.0*(m*x1 - y1));
		
    if (m > 0.0) {
      if (y1 < y2) { 
	ystart = (double) y1;
	yend = (double) y2;
      }
      else {
	ystart = (double) y2;
	yend = (double) y1;
      }
    }
    else {
      if (y1 > y2){
	ystart = (double) y1;
	yend = (double) y2;
      }
      else {
	ystart = (double) y2;
	yend = (double) y1;
      }
    }
		
    tempY = (double) ystart;
    boolean hitEdge = false;
		
    if (tempY > yend) {
      while(tempY >= yend) {
	tempX = (double) ((tempY-b)/m);
	if(box.inside((int)tempX,(int)tempY)) {
	  hitEdge = true;
	  break;
	}
	tempY = tempY + m;
      }
    }    
    else {
      while(tempY <= yend) {
	tempX = (double) ((tempY-b)/m);
	if(box.inside((int)tempX,(int)tempY)) {
	  hitEdge = true;
	  break;
	}
	tempY = tempY + m;
      }
    }
    return hitEdge;
  }
	
  private boolean checkEdgeHitx(int x1, int x2, int y, Rectangle box) {
    int beginx, endx;
    boolean hitEdge = false;
		
    if (x1 < x2) {
      beginx = x1;
      endx = x2;
    }
    else {
      beginx = x2;
      endx = x1;
    }
		
    for(int i=beginx; i<=endx; i++) {
      if(box.inside(i,y)) {
	hitEdge = true;
	break;
      }
    }
    return hitEdge;
  }
	
  private boolean checkEdgeHity(int y1, int y2, int x, Rectangle box) {
    int beginy, endy;
    boolean hitEdge = false;
		
    if (y1 < y2) {
      beginy = y1;
      endy = y2;
    }
    else {
      beginy = y2;
      endy = y1;
    }
		
    for(int i=beginy; i<=endy; i++) {
      if(box.inside(x,i)) {
	hitEdge = true;
	break;
      }
    }
    return hitEdge;
  }
	
  /**
   * This is coming from a toolbar request. Must check if a single, node
   * is selected, error otherwise.
   */
  public void requestLink() {
		
    CpeNode selectedNode = null;
    selectedNode = getSingleSelectedNode();
    if (selectedNode != null) {
      m_panel.m_hitNode = selectedNode;
      m_linkType = 0;
      addLink(selectedNode, true);
    }
  }

  /**
   * This is coming from a toolbar request. Must check if a single, node
   * is selected, error otherwise.
   */
  public void requestPos() {
		
    CpeNode selectedNode = null;
    selectedNode = getSingleSelectedNode();
    if (selectedNode != null) {
      m_panel.m_hitNode = selectedNode;
      m_linkType = 1;
      addLink(selectedNode, true);
    }
  }

  /**
   * This is coming from a toolbar request. Must check if a single, node
   * is selected, error otherwise.
   */
  public void requestNeg() {
		
    CpeNode selectedNode = null;
    selectedNode = getSingleSelectedNode();
    if (selectedNode != null) {
      m_panel.m_hitNode = selectedNode;
      m_linkType = 2;
      addLink(selectedNode, true);
    }
  }
	
  /**
   * This is coming from a toolbar request. Must check if a single, node
   * is selected, error otherwise.
   */
  public void requestNewEqual() {
		
    CpeNode selectedNode = null;
    selectedNode = getSingleSelectedNode();
    if (selectedNode != null) {
      m_panel.m_hitNode = selectedNode;
      addEqLink(selectedNode, true);
    }
  }
	
	
  /**
   * This is coming from a toolbar request. Must check if a single, node
   * is selected, error otherwise.
   */
  public void requestExpansion() {
		
    CpeNode selectedNode = null;
    selectedNode = getSingleSelectedNode();
    if (selectedNode != null) {
      m_panel.m_hitNode = selectedNode;
      showExpansion(selectedNode);
    }
  }
	
  /**
   * This is coming from a toolbar request.
   * Show the process containing the expanded node.
   */
  public void requestUp() {
		
    //if (topLevelFlag == false)
    //m_panel.mainPanel.showExpansion(expands, m_panel);
  }
	
  /**
   * This is coming from a toolbar request. Must check if a single, node
   * is selected, if not, display properties for process.
   */
  public void requestProperties() {
    CpeNode tmpNode = null;
    CpeNode selectedNode = null;
    int count=0;
		
    for  (Enumeration  e  =  m_nodes.elements() ;
	  e.hasMoreElements()  ;)  {	
      tmpNode = (CpeNode) e.nextElement();
      if (tmpNode.m_selected == true) {
	selectedNode = tmpNode;
	count++;
      }
      if (count > 1) {
	JOptionPane.showMessageDialog(Cpe.sharedInstance().getFrame(), 
				      "You must select only one node.",
				      "CPE Message", 
				      JOptionPane.WARNING_MESSAGE);
	break;
      }
			
    }
		
    if (count == 0) {
      showProcessPropertyDialog();
    }
    else if (count == 1) {
      m_panel.m_hitNode = selectedNode;
      showNodePropertyDialog();
    }
  }
	
  public CpeNode getSingleSelectedNode() {
		
    CpeNode tmpNode = null;
    CpeNode selectedNode = null;
    int count=0;
		
    for  (Enumeration  e  =  m_nodes.elements() ;
	  e.hasMoreElements()  ;)  {	
      tmpNode = (CpeNode) e.nextElement();
      if (tmpNode.m_selected == true) {
	selectedNode = tmpNode;
	count++;
      }
      if (count > 1) {
	JOptionPane.showMessageDialog(Cpe.sharedInstance().getFrame(), 
				      "You must select only one node.",
				      "CPE Message", 
				      JOptionPane.WARNING_MESSAGE);
	break;
      }
			
    }
		
    if (count == 0) {
      JOptionPane.showMessageDialog(Cpe.sharedInstance().getFrame(), 
				    "You must first select a node.",
				    "CPE Message", 
				    JOptionPane.WARNING_MESSAGE);
      selectedNode = null;
      return selectedNode;
    }
    else if (count == 1) {
      return  selectedNode;
    }
    selectedNode = null;
    return selectedNode;
  }
	
  public void addLink(CpeNode node, boolean fromToolbar) {
    m_linkDrawing = true;
		
    if(node.m_key.equals("Start")) {
      //start can only link from its end half
      m_linkBegin = false;
    }else if(node.m_key.equals("Finish")) {
      //finish can only link from its begin half
      m_linkBegin = true;
    }else if (fromToolbar) {
      //toolbar initiated actions will always assume end half
      m_linkBegin = false;
    }else {
      m_linkBegin = node.isBegin(popupPoint);
    }

    m_linkToX = node.m_x;
    m_linkToY = node.m_y;
  }
	
  public void addEqLink(CpeNode node, boolean fromToolbar) {
    m_linkDrawing = true;
		
    if(node.m_key.equals("Start")) {
      //start can only link from its end half
      m_linkBegin = false;
    }else if(node.m_key.equals("Finish")) {
      //finish can only link from its begin half
      m_linkBegin = true;
    } else if (fromToolbar) {
      //toolbar initiated actions will always assume end half
      m_linkBegin = false;
    } else {
      m_linkBegin = node.isBegin(popupPoint);
    }
    m_linkType = 0;
    m_linkToX = node.m_x;
    m_linkToY = node.m_y;
  }
	
  public void showNodePropertyDialog() {
    if (m_sheet == null) {
      m_sheet = new CpeRatNodePropertyDialog(m_panel, 
					     "Node Property Sheet...");
    }
    m_sheet.setLocation(Cpe.sharedInstance().getCenteringPoint(m_sheet));
    m_sheet.show();
  }
	
  public void showProcessPropertyDialog() {
    if (processSheet == null) {
      processSheet = new CpeRatProcPropertyDialog(m_panel, 
						  "Process Property Sheet...");
    }
    processSheet.setLocation
      (Cpe.sharedInstance().getCenteringPoint(processSheet));
    processSheet.show();
  }
	
  public void alignTop() {
    Vector selNodes = new Vector();
    CpeNode tempNode;
    int top = (int) Integer.MAX_VALUE;
		
    for  (Enumeration e =  m_nodes.elements() ; 
	  e.hasMoreElements()  ;)  {	
      tempNode = (CpeNode) e.nextElement();
      if (tempNode.m_selected){
	selNodes.addElement(tempNode);
	if (tempNode.m_y < top) top = tempNode.m_y;
      }
    }
    for  (Enumeration  e2  =  selNodes.elements() ;
	  e2.hasMoreElements()  ;)  {	
      tempNode = (CpeNode) e2.nextElement();
      tempNode.m_y = top;
    }
    selNodes = null;
  }
	
  public void alignBottom() {
    Vector selNodes = new Vector();
    CpeNode tempNode;
    int bottom = 0;
		
    for  (Enumeration e =  m_nodes.elements() ; 
	  e.hasMoreElements()  ;)  {	
      tempNode = (CpeNode) e.nextElement();
      if (tempNode.m_selected) {
	selNodes.addElement(tempNode);
	if (tempNode.m_y > bottom) bottom = tempNode.m_y;
      }
    }
    for  (Enumeration  e2  =  selNodes.elements() ;
	  e2.hasMoreElements()  ;)  {	
      tempNode = (CpeNode) e2.nextElement();
      tempNode.m_y = bottom;
    }
    selNodes = null;
  }
	
  public void alignLeft() {
    Vector selNodes = new Vector();
    CpeNode tempNode;
    int left = (int) Integer.MAX_VALUE;
		
    for  (Enumeration e =  m_nodes.elements() ; 
	  e.hasMoreElements()  ;)  {	
      tempNode = (CpeNode) e.nextElement();
      if (tempNode.m_selected) {
	selNodes.addElement(tempNode);
	if (tempNode.m_rect.x < left) left = tempNode.m_rect.x;
      }
    }
    FontMetrics fm = m_panel.getFontMetrics(m_panel.getFont());
    for  (Enumeration  e2  =  selNodes.elements() ;
	  e2.hasMoreElements()  ;)  {	
      tempNode = (CpeNode) e2.nextElement();
      tempNode.setLeftPosition(left,fm);
    }
    selNodes = null;
  }
	
  public void alignRight() {
    Vector selNodes = new Vector();
    CpeNode tempNode;
    int right = 0;
		
    for  (Enumeration e =  m_nodes.elements() ; 
	  e.hasMoreElements()  ;)  {	
      tempNode = (CpeNode) e.nextElement();
      if (tempNode.m_selected) {
	selNodes.addElement(tempNode);
	if ((tempNode.m_rect.x + tempNode.m_rect.width)  > right) {
	  right = tempNode.m_rect.x + tempNode.m_rect.width;
	}
      }
    }
    FontMetrics fm = m_panel.getFontMetrics(m_panel.getFont());
    for  (Enumeration  e2  =  selNodes.elements() ;
	  e2.hasMoreElements()  ;)  {	
      tempNode = (CpeNode) e2.nextElement();
      tempNode.setRightPosition(right, fm);
    }
    selNodes = null;
  }
	
  public void alignCenter() {
    Vector selNodes = new Vector();
    CpeNode tempNode;
    int avg = 0;
    int count = 0;
		
    for  (Enumeration e =  m_nodes.elements() ; 
	  e.hasMoreElements()  ;)  {	
      tempNode = (CpeNode) e.nextElement();
      if (tempNode.m_selected) {
	selNodes.addElement(tempNode);
	avg = avg + tempNode.m_x;
	count++;
      }
    }
    if (count != 0) avg = (int) avg/count;
    for  (Enumeration  e2  =  selNodes.elements() ;
	  e2.hasMoreElements()  ;)  {	
      tempNode = (CpeNode) e2.nextElement();
      tempNode.m_x = avg;
    }
  }
	
  public void addExpansion(CpeNode node) {
		
    if (node.hasExpansion) {
      JOptionPane.showMessageDialog
	(Cpe.sharedInstance().getFrame(), 
	 "You must first remove the existing expansion.",
	 "CPE Message", 
	 JOptionPane.WARNING_MESSAGE);
    } else {
      node.hasExpansion = true;
      m_panel.repaint();
    }
  }
	
  public void delExpansion(CpeNode node) {
		
    JOptionPane.showMessageDialog
      (Cpe.sharedInstance().getFrame(), 
       "Deleting an expansion is not supported yet",
       "CPE Message", 
       JOptionPane.WARNING_MESSAGE);
  }
  
  public void showExpansion(CpeNode node) {
		
    if (node.hasExpansion)
      m_panel.repaint();
  }
	
  public void setKey(String key) {
		
    this.key = key;
    if (label.equals("") == true) { 
      this.label = key;
    }
  }
	
  public void setLabel(String label) {
		
    //need to relabel the frame and possibly the tab
    if (this.label.equals("") == false) { 
      m_panel.mainPanel.renameProcess(m_panel, 
				      this.label, label, topLevelFlag);
    }
		
    this.label = label;
  }
	
  public void addText(String label) {
    m_text.addElement(new CpeText(label, popupPoint.x, popupPoint.y));    
  }
	
  public void setStartFinishFlag(boolean flag) {
    String oldLeftBoundString = null;
    String oldRightBoundString = null;
    String newLeftBoundString = null;
    String newRightBoundString = null;
		
    //only do something when there is a change
    if (this.startFinishFlag == flag)
      return;
        
    this.startFinishFlag = flag;
		
    if (this.startFinishFlag) {
      oldLeftBoundString = BEGIN_STRING;
      oldRightBoundString = END_STRING;
      newLeftBoundString = START_STRING;
      newRightBoundString = FINISH_STRING;
    } else {
      oldLeftBoundString = START_STRING;
      oldRightBoundString = FINISH_STRING;
      newLeftBoundString = BEGIN_STRING;
      newRightBoundString = END_STRING;
    }
		
    CpeNode tempNode;
    for  (Enumeration e =  m_nodes.elements() ; 
	  e.hasMoreElements()  ;)  {	
      tempNode = (CpeNode) e.nextElement();
      if (tempNode.m_key == oldLeftBoundString) {
	tempNode.m_key = newLeftBoundString;
	tempNode.m_lbl = newLeftBoundString;
      }
      if (tempNode.m_key == oldRightBoundString) {
	tempNode.m_key = newRightBoundString;
	tempNode.m_lbl = newRightBoundString;
      }
    }
    m_panel.repaint();
  }
	
  // class used to determine if consistent network exists. 
  class Vertex {
    public String name = "";
    public int inDegree = 0;
		
    Vertex(String Name) {
      name = Name;
    }	
  }
	
  String getLinear() {
		
    //basically, this collapses all the equal points to single points
    //and then topologically sorts them. If a solution cannot be
    //found, it returns "No solution!".
		
    //1st we need to collapse all points and assign them 
    //a number correponding to their entry in a vertex array
    Vector v = collapsePoints();
    v = calculateInDegree(v); 
		
    //debug printing 
    /*  
	Vertex tempVert1;
	for  (Enumeration e =  v.elements();e.hasMoreElements()  ;)  {	
	tempVert1 = (Vertex) e.nextElement();
	System.out.println(tempVert1.name+" deg: "+tempVert1.inDegree);
	}   */
		
    int[] algDegree;  
    String ordering = "";
    algDegree = new int[v.size()];
    int x=0;
    Vertex tempVert;
		
    for  (Enumeration e =  v.elements();e.hasMoreElements()  ;)  {	
      tempVert = (Vertex) e.nextElement();
      algDegree[x] = tempVert.inDegree;
      x++;
    }
    int vertexCount = 0;
    int i;
    while (vertexCount< v.size()) {
      i=0;
      for (;i<v.size() && algDegree[i]!=0;i++);
      if(i==v.size()) {
	//System.out.println("No solution!");
	return "No Solution!";
      }
      //got a zero degree, remove it, add it to the ordering
      algDegree[i]--;
      if (vertexCount == 0) {
	ordering= ((Vertex)v.elementAt(i)).name;
      }
      else {
	ordering=
	  ordering+"\n-> "+
	  ((Vertex)v.elementAt(i)).name;
      }

      //now we need to decrease the degree on each of the out
      //going edges from the node(s) which were removed.
			
      //this section checks the explicit orderings
      CpeEdge tempEdge;
      for  (Enumeration e = m_edges.elements();e.hasMoreElements() ;) {	
	tempEdge = (CpeEdge) e.nextElement();
	boolean found = false;
	//only check "before" edges
	if (("CpeEqEdge".equals(tempEdge.getClass().getName())) 
	    == false) {
	  if (tempEdge.fromPosition /*from begin*/) { 
	    if (tempEdge.m_from.beginNumber == i)
	      found = true; 
	  }
	  else
	    if(tempEdge.m_from.endNumber == i)
	      found = true;
	}
	if (found == true) {
	  if (tempEdge.toPosition == true /*to Begin*/) {
	    algDegree[tempEdge.m_to.beginNumber]--;
	  }
	  else {
	    algDegree[tempEdge.m_to.endNumber]--;
	  }
	}
      }
			
      //this section removes the implicit begin-end degree within a node
      // if necessary.
      CpeNode tempNode;
      for  (Enumeration e = m_nodes.elements();e.hasMoreElements() ;) {	
	tempNode = (CpeNode) e.nextElement();
	if (i == tempNode.beginNumber) {
	  algDegree[tempNode.endNumber]--;
	} 
      }
			
      vertexCount++;
    }
    //System.out.println("Solution: "+ordering);
    return ordering;
  }
	
  //collapse points is the 1st step in the topological sort It returns
  //a new vector of collapsed points.
  private Vector collapsePoints() {
		
    Vector v = new Vector();
    Vertex tempVert;
    CpeNode tempNode;
    CpeNode tempNode1;
    CpeEdge tempEdge;
    int pointCounter = 0;
		
    //initalize flags and values
    for  (Enumeration e =  m_nodes.elements() ; 
	  e.hasMoreElements()  ;)  {	
      tempNode = (CpeNode) e.nextElement();
      tempNode.visitedBegin = false;
      tempNode.visitedEnd = false;
      tempNode. beginNumber = 0;
      tempNode.endNumber = 0;
    }
		
    //visit each node's points
    for  (Enumeration e =  m_nodes.elements();e.hasMoreElements()  ;)  {
      tempNode = (CpeNode) e.nextElement();
			
      if (tempNode.visitedBegin == false) {
	//haven't been here, mark it, create a vertex
	tempNode.visitedBegin = true;
	tempNode.beginNumber = pointCounter++;
	tempVert = new Vertex(tempNode.m_lbl+".beg");
				
	//now need to look for equal links which will 
	//collapse into this point
	for  (Enumeration e1 =  m_edges.elements() ; e1.hasMoreElements() ;) {
	  tempEdge = (CpeEdge) e1.nextElement();
	  if ("CpeEqEdge".equals(tempEdge.getClass().getName())) {
	    //checking the "to" side
	    if ((tempEdge.m_to.m_key == tempNode.m_key) &&
		(tempEdge.toPosition == true /* it is the begin half*/)){
	      //found it on one side of an eq relation.
	      //mark the other node's point
	      if (tempEdge.fromPosition == true 
		  /*it is to the other begin*/) {
		tempEdge.m_from.visitedBegin = true;
		tempEdge.m_from.beginNumber = pointCounter - 1;
		tempVert.name = tempVert.name + "/" + tempEdge.m_from.m_key +
		  ".beg";
	      }
	      else {
		tempEdge.m_from.visitedEnd = true;
		tempEdge.m_from.endNumber = pointCounter - 1;
		tempVert.name = tempVert.name + "/" + tempEdge.m_from.m_key +
		  ".end";
	      }
							
	    }
	    //checking the "from" side
	    if ((tempEdge.m_from.m_key == tempNode.m_key) &&
		(tempEdge.fromPosition == true /* it is the begin half*/)){
	      //found it on one side of an eq relation.
	      //mark the other node's point
	      if (tempEdge.toPosition == true /*it is to the other begin*/) {
		tempEdge.m_to.visitedBegin = true;
		tempEdge.m_to.beginNumber = pointCounter - 1;
		tempVert.name = tempVert.name + "/" + tempEdge.m_to.m_key +
		  ".beg";
	      }
	      else {
		tempEdge.m_to.visitedEnd = true;
		tempEdge.m_to.endNumber = pointCounter - 1;
		tempVert.name = tempVert.name + "/" + tempEdge.m_to.m_key +
		  ".end";
	      }
							
	    }
	  }
	}
				
	//add tempVertex to the vector
	v.addElement(tempVert);
      }
			
      //checking end half now
      if (tempNode.visitedEnd == false) {
	//haven't been here, mark it, create a vertex
	tempNode.visitedEnd = true;
	tempNode.endNumber = pointCounter++;
	tempVert = new Vertex(tempNode.m_lbl+".end");
				
	//now need to look for equal links which will 
	//collapse into this point
	for  (Enumeration e1 =  m_edges.elements() ; e1.hasMoreElements() ;) {
	  tempEdge = (CpeEdge) e1.nextElement();
	  if ("CpeEqEdge".equals(tempEdge.getClass().getName())) {
	    //checking the "to" side
	    if ((tempEdge.m_to.m_key == tempNode.m_key) &&
		(tempEdge.toPosition == false /* it is the end half*/)){
	      //found it on one side of an eq relation.
	      //mark the other node's point
	      if (tempEdge.fromPosition == true /*it is to the other begin*/) {
		tempEdge.m_from.visitedBegin = true;
		tempEdge.m_from.beginNumber = pointCounter - 1;
		tempVert.name = tempVert.name + "/" + tempEdge.m_from.m_key +
		  ".beg";
	      }
	      else {
		tempEdge.m_from.visitedEnd = true;
		tempEdge.m_from.endNumber = pointCounter - 1;
		tempVert.name = tempVert.name + "/" + tempEdge.m_from.m_key +
		  ".end";
	      }
							
	    }
	    //checking the "from" side
	    if ((tempEdge.m_from.m_key == tempNode.m_key) &&
		(tempEdge.fromPosition == false /* it is the end half*/)){
	      //found it on one side of an eq relation.
	      //mark the other node's point
	      if (tempEdge.toPosition == true /*it is to the other begin*/) {
		tempEdge.m_to.visitedBegin = true;
		tempEdge.m_to.beginNumber = pointCounter - 1;
		tempVert.name = tempVert.name + "/" + tempEdge.m_to.m_key +
		  ".beg";
	      }
	      else {
		tempEdge.m_to.visitedEnd = true;
		tempEdge.m_to.endNumber = pointCounter - 1;
		tempVert.name = tempVert.name + "/" + tempEdge.m_to.m_key +
		  ".end";
	      }
							
	    }
	  }
	}
				
	//add tempVertex to the vector, increase the indegree by one since
	//we know that it is at least one relation to the begin.
	tempVert.inDegree++;
	v.addElement(tempVert);
      }
			
			
    }
    return v;
  }

  //this routine runs thru the edge vector accumulating the indegree
  // values for the collapsed verticies
  Vector calculateInDegree(Vector v) {
    CpeEdge tempEdge;
    for  (Enumeration e = m_edges.elements();e.hasMoreElements()  ;)  {	
      tempEdge = (CpeEdge) e.nextElement();
      //only check "before" edges
      if (("CpeEqEdge".equals(tempEdge.getClass().getName())) 
	  == false) {
	if (tempEdge.toPosition == true /*to Begin*/) {
	  ((Vertex)v.elementAt(tempEdge.m_to.beginNumber)).inDegree++;
	}
	else {
	  ((Vertex)v.elementAt(tempEdge.m_to.endNumber)).inDegree++;
	}
      } 
    }   
    return v;
  }

}
